import UIKit
import Alamofire

// Protocol for Android-like scroll-to-collapse behavior
protocol TransactionsViewDelegate: AnyObject {
    func shouldCollapseBottomSheet()
    func shouldExpandBottomSheet()
}

class TransactionsView: UIViewController, UICollectionViewDelegate, UICollectionViewDataSource, UICollectionViewDelegateFlowLayout {

    @IBOutlet weak var transactionCollectionView: UICollectionView!
    @IBOutlet weak var indicator: UIView!
    @IBOutlet weak var progressIndicator: UIActivityIndicatorView!
    
    // Android-like scroll-to-collapse delegate
    weak var delegate: TransactionsViewDelegate?
    
    // Programmatic indicator to avoid storyboard constraint issues
    private lazy var programmaticIndicator: UIView = {
        let indicatorView = UIView()
        indicatorView.backgroundColor = SDKVariableSetting.isDarkMode == true ? UIColor(hex:SDKVariableSetting.indicatorDark) :UIColor(hex:SDKVariableSetting.indicatorLight)
        indicatorView.layer.cornerRadius = 4
//        indicatorView.alpha = 0.6
        indicatorView.translatesAutoresizingMaskIntoConstraints = false
        return indicatorView
    }()
    
    private var transactions: [DWalletTransaction] = []
    var accountId: String = ""
    
    // Android-like scroll behavior control
    private var isBottomSheetCollapsed: Bool = true
    private var isAtTopOfContent: Bool = false
    private var hasScrolledPastTop: Bool = false
    
    // Enhanced scroll-to-collapse control
    private var collapseScrollLockActive: Bool = false
    private var hasUserScrolledAwayFromTop: Bool = false

    private var currentPage = 1
    private let pageSize = 10
    private var isLastPage = false
    var language:String = ""
    private var lastFetchTime: Date?

    private var isLoading: Bool = false {
        didSet {
            updateLoadingState()
        }
    }

    override func viewDidLoad() {
        super.viewDidLoad()

        view.backgroundColor = DefaultAppearance.shared.cardColor
        
        getSharePref()
        
        setUpCollectionViewCell()
        setupProgrammaticIndicator()

        // Hide IBOutlet indicator to prevent conflicts
        progressIndicator.isHidden = true

        // Add footer loading view
        view.addSubview(footerLoadingView)
        NSLayoutConstraint.activate([
            footerLoadingView.centerXAnchor.constraint(equalTo: view.centerXAnchor),
            footerLoadingView.bottomAnchor.constraint(equalTo: view.bottomAnchor, constant: -20)
        ])

        // Add empty data label with top alignment
        view.addSubview(emptyDataLabel)
        emptyDataLabel.translatesAutoresizingMaskIntoConstraints = false
        NSLayoutConstraint.activate([
            emptyDataLabel.centerXAnchor.constraint(equalTo: view.centerXAnchor),
            emptyDataLabel.topAnchor.constraint(equalTo: view.safeAreaLayoutGuide.topAnchor, constant: 40), // Add padding from top
            emptyDataLabel.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 20),
            emptyDataLabel.trailingAnchor.constraint(equalTo: view.trailingAnchor, constant: -20)
        ])

        fetchTransactions() // Fetch transactions when the view loads
    }

    // Recalculate insets whenever layout changes (e.g., bottom sheet height adjustment)
    override func viewDidLayoutSubviews() {
        super.viewDidLayoutSubviews()
        updateCollectionInsets()
    }
    
    
    private func startLoading(){
        // Only use footer loading view for consistency
        // Hide the IBOutlet indicator to avoid conflicts
        progressIndicator.stopAnimating()
        progressIndicator.isHidden = true
    }
    
    private func stopLoading(){
        // Only manage footer loading view
        progressIndicator.stopAnimating()
        progressIndicator.isHidden = true
    }
    
    
    private func getSharePref(){
        if let languageCode = SharedPreferenceManager.getString(forKey: SharePrefKey.lanuageCode.rawValue){
            language = languageCode
        }else{
            language = "km"
        }
    }

    private func setUpCollectionViewCell() {
        let nibName = UINib(nibName: "TransactionViewCell", bundle: B24PaymentSdkHelper.frameworkBundle())
        self.transactionCollectionView.backgroundColor = DefaultAppearance.shared.cardColor
        self.transactionCollectionView.register(nibName, forCellWithReuseIdentifier: "TransactionViewCell")
        self.transactionCollectionView.delegate = self
        self.transactionCollectionView.dataSource = self
    }

    private func setupProgrammaticIndicator() {
        // Hide the IBOutlet indicator to avoid conflicts
        indicator.isHidden = true
        
        // Add our programmatic indicator
        view.addSubview(programmaticIndicator)
        
        // Set up constraints for fixed size and positioning
        NSLayoutConstraint.activate([
            // Fixed size constraints - never changes
            programmaticIndicator.widthAnchor.constraint(equalToConstant: 40),
            programmaticIndicator.heightAnchor.constraint(equalToConstant: 5),
            
            // Positioning constraints
            programmaticIndicator.centerXAnchor.constraint(equalTo: view.centerXAnchor),
            programmaticIndicator.topAnchor.constraint(equalTo: view.topAnchor, constant: 8)
        ])
    }

    // MARK: - Adaptive Insets
    private func updateCollectionInsets() {
        view.layoutIfNeeded()
        
        // Calculate proper bottom inset to ensure last item is properly visible
        let footerHeight: CGFloat = footerLoadingView.isHidden ? 0 : 44  // Height for loading indicator
        let safeAreaBottom: CGFloat = view.safeAreaInsets.bottom
        let basicPadding: CGFloat = 20  // Additional padding to ensure last item is fully visible
        
        // Ensure the last item has enough space to be fully visible above any bottom content
        let bottomInset = footerHeight + basicPadding + safeAreaBottom
        
        let insets = UIEdgeInsets(top: 8, left: 0, bottom: bottomInset, right: 0)
        transactionCollectionView.contentInset = insets
        transactionCollectionView.scrollIndicatorInsets = insets
        
        // Force layout update to apply changes immediately
        transactionCollectionView.setNeedsLayout()
        transactionCollectionView.layoutIfNeeded()
    }

    // Parent calls after resizing sheet height
    func refreshInsets() { updateCollectionInsets() }
    
    // MARK: - Android-like Scroll Control
    func setBottomSheetState(collapsed: Bool) {
        isBottomSheetCollapsed = collapsed
        
        // Reset all scroll state when state changes
        resetScrollState()
        
        // Android behavior: Disable scrolling when collapsed (medium state)
        // Enable scrolling when expanded (large state)
        transactionCollectionView.isScrollEnabled = !collapsed
        
        if collapsed {
            // When collapsing: Always reset to top position immediately
            // This ensures that when re-expanded, user can scroll properly
            transactionCollectionView.setContentOffset(.zero, animated: false)
            isAtTopOfContent = true
        } else {
            // When expanding: Enable scrolling and ensure we're at top
            // Reset to top position to provide consistent behavior
            DispatchQueue.main.async {
                self.transactionCollectionView.setContentOffset(.zero, animated: false)
                self.isAtTopOfContent = true
                // Small delay to ensure the UI has updated before enabling proper scroll behavior
                DispatchQueue.main.asyncAfter(deadline: .now() + 0.1) {
                    self.transactionCollectionView.isScrollEnabled = true
                }
            }
        }
    }

    // MARK: - Grip / Indicator adaptive appearance
    // progress: 0 (collapsed) -> 1 (expanded)
    func updateGripAppearance(progress: CGFloat) {
        // Update only the opacity of our programmatic indicator
        // Size remains fixed at 40x5 points always
        let baseAlpha: CGFloat = 1
        let expandedAlpha: CGFloat = 1
        programmaticIndicator.alpha = baseAlpha + (expandedAlpha - baseAlpha) * progress
    }


    func scrollViewDidScroll(_ scrollView: UIScrollView) {
        let offsetY = scrollView.contentOffset.y
        
        // Only handle collapse behavior when bottom sheet is expanded
        if !isBottomSheetCollapsed {
            let isCurrentlyAtTop = offsetY <= 5 // Small threshold for "at top"
            
            // Update our top tracking
            if isCurrentlyAtTop {
                isAtTopOfContent = true
            } else if offsetY > 20 {
                isAtTopOfContent = false
                hasUserScrolledAwayFromTop = true
            }
            
            // Android-like collapse behavior:
            // Allow collapse when user is at top AND has previously scrolled away
            // This prevents accidental collapses on initial expansion
            if isAtTopOfContent && hasUserScrolledAwayFromTop && offsetY < -30 {
                delegate?.shouldCollapseBottomSheet()
                resetScrollState()
                return
            }
        }
        
        // Pagination: Load more content when near bottom
        guard !isLoading, !isLastPage else { return }
        
        let contentHeight = scrollView.contentSize.height
        let frameHeight = scrollView.frame.size.height
        let threshold: CGFloat = 120
        
        if offsetY > contentHeight - frameHeight - threshold {
            fetchTransactions()
        }
    }
    
    private func resetScrollState() {
        isAtTopOfContent = true // Start assuming we're at top
        hasScrolledPastTop = false
        collapseScrollLockActive = false
        hasUserScrolledAwayFromTop = false
        
        // Ensure collection view is enabled and properly positioned
        if !isBottomSheetCollapsed {
            transactionCollectionView.isScrollEnabled = true
        }
    }

    private let footerLoadingView: UIActivityIndicatorView = {
        let indicator = UIActivityIndicatorView(style: .medium)
        indicator.hidesWhenStopped = true
        indicator.color = DefaultAppearance.shared.primaryColor
        indicator.translatesAutoresizingMaskIntoConstraints = false
        return indicator
    }()

    private lazy var emptyDataLabel: UIView = {
            let containerView = UIView()
            containerView.isHidden = true
            containerView.translatesAutoresizingMaskIntoConstraints = false

            // Create exclamation icon image view
            let iconImageView = UIImageView()
            iconImageView.translatesAutoresizingMaskIntoConstraints = false
            iconImageView.image = UIImage(systemName: "clock.arrow.circlepath")
            iconImageView.tintColor = DefaultAppearance.shared.secondaryLabelColor
            iconImageView.contentMode = .scaleAspectFit

            // Create label
            let label = UILabel()
            label.translatesAutoresizingMaskIntoConstraints = false
            label.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.no_tran_history.rawValue)
            label.textColor = DefaultAppearance.shared.secondaryLabelColor
        label.font = FontManager.shared.mediumFont(forLanguage: "en" ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.contentText)
            label.textAlignment = .center
            label.numberOfLines = 0
       

            // Add subviews
            containerView.addSubview(iconImageView)
            containerView.addSubview(label)

            // Setup constraints
            NSLayoutConstraint.activate([
                iconImageView.centerXAnchor.constraint(equalTo: containerView.centerXAnchor),
                iconImageView.topAnchor.constraint(equalTo: containerView.topAnchor),
                iconImageView.widthAnchor.constraint(equalToConstant: 40),
                iconImageView.heightAnchor.constraint(equalToConstant: 40),

                label.topAnchor.constraint(equalTo: iconImageView.bottomAnchor, constant: 16),
                label.leadingAnchor.constraint(equalTo: containerView.leadingAnchor, constant: 20),
                label.trailingAnchor.constraint(equalTo: containerView.trailingAnchor, constant: -20),
                label.bottomAnchor.constraint(equalTo: containerView.bottomAnchor)
            ])

            return containerView
        }()

    // MARK: - Fetch Transactions
    private func fetchTransactions() {
        guard !isLoading, !isLastPage else { return }
        
        // Debounce: prevent rapid successive calls
        let now = Date()
        if let lastFetch = lastFetchTime, now.timeIntervalSince(lastFetch) < 0.5 {
            return
        }
        lastFetchTime = now
        
        // If this is the first page, reset the state
        if currentPage == 1 {
            transactions.removeAll()
            isLastPage = false
            // Hide empty data label during initial loading
            emptyDataLabel.isHidden = true
        }
        
        isLoading = true
        footerLoadingView.startAnimating()
        updateCollectionInsets()
        
        // Don't call startLoading() - causes conflict with footerLoadingView

        let url = APIManager.merchantApiUrl().appendingPathComponent("instantpaymentsdk/payment_method/transactions")
        let parameters: [String: Any] = [
            "id": accountId,
            "pagination": [
                "current_page": currentPage,
                "page_size": pageSize
            ]
        ]

        AF.request(
            url,
            method: .post,
            parameters: parameters,
            encoding: JSONEncoding.default,
            headers: APIManager.initHeader()
        )
        .validate()
        .responseDecodable(of: DWalletTransactionResponseModel.self) { [weak self] response in
            guard let self = self else { return }
            self.isLoading = false
            self.footerLoadingView.stopAnimating()
            self.updateCollectionInsets()

            switch response.result {
            case .success(let responseModel):
                if responseModel.code == StatusCode.succcess.rawValue {
                    // Don't call stopLoading() - already handled by footerLoadingView.stopAnimating()
                    
                    // Filter out any transactions that already exist (prevent duplicates)
                    let existingTransactionIDs = Set(self.transactions.map { $0.tranID })
                    let newTransactions = responseModel.data.transactions.filter { !existingTransactionIDs.contains($0.tranID) }
                    
                    print("DEBUG: Page \(self.currentPage), Got \(responseModel.data.transactions.count) transactions, \(newTransactions.count) are new")
                    
                    self.transactions.append(contentsOf: newTransactions)
                    self.isLastPage = responseModel.data.pagination.hasNextPage == false
                    self.transactionCollectionView.reloadData()
                    self.updateCollectionInsets()
                    self.emptyDataLabel.isHidden = !self.transactions.isEmpty
                    self.currentPage += 1
                } else {
//                    self.showError(message: responseModel.message)
                    let message = self.language == "en" ? responseModel.message : responseModel.messageKh
                    B24PaymentSdkHelper.errorSnackbar(view: self.view,message: message )
                    // Show empty data label if no transactions and it's the first page
                    if self.currentPage == 1 {
                        self.emptyDataLabel.isHidden = false
                    }
                }
            case .failure(let error):
                print("=====>\(error)")
                // Don't call stopLoading() - already handled by footerLoadingView.stopAnimating()
                //self.showError(message: "Network error: \(error.localizedDescription)")
                B24PaymentSdkHelper.errorSnackbar(
                    view:self.view,
                                message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                                                    forBottomSheet: false)
                // Show empty data label if no transactions and it's the first page
                if self.currentPage == 1 {
                    self.emptyDataLabel.isHidden = false
                }
            }
        }
    }

    // MARK: - Public Methods
    func refreshTransactions() {
        // Reset pagination state and reload from beginning
        currentPage = 1
        isLastPage = false
        isLoading = false
        lastFetchTime = nil
        transactions.removeAll()
        transactionCollectionView.reloadData()
        
        // Reset scroll state and position
        resetScrollState()
        transactionCollectionView.setContentOffset(.zero, animated: false)
        
        fetchTransactions()
    }

    // MARK: - Loading State
    private func updateLoadingState() {
        DispatchQueue.main.async {
            // Only manage footer loading view to avoid conflicts
            self.footerLoadingView.isHidden = !self.isLoading
            
            // Only show empty data label when not loading and no transactions
            if !self.isLoading {
                self.emptyDataLabel.isHidden = !self.transactions.isEmpty
            } else {
                // Always hide empty data label when loading
                self.emptyDataLabel.isHidden = true
            }
        }
    }

    // MARK: - Error Handling
    private func showError(message: String) {
        let alert = UIAlertController(title: "Error", message: message, preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "OK", style: .default))
        present(alert, animated: true)
    }

    // MARK: - Collection View Data Source
    func numberOfSections(in collectionView: UICollectionView) -> Int {
        return 1
    }

    func collectionView(_ collectionView: UICollectionView, numberOfItemsInSection section: Int) -> Int {
        return transactions.count
    }

    func collectionView(_ collectionView: UICollectionView, cellForItemAt indexPath: IndexPath) -> UICollectionViewCell {
        let cell = transactionCollectionView.dequeueReusableCell(withReuseIdentifier: "TransactionViewCell", for: indexPath) as! TransactionViewCell
        let transaction = transactions[indexPath.item]
        configureTransactionCell(cell, with: transaction)
        return cell
    }

    private func configureTransactionCell(_ cell: TransactionViewCell, with transaction: DWalletTransaction) {
        cell.container.backgroundColor = DefaultAppearance.shared.cardColor
        
        cell.line.backgroundColor = DefaultAppearance.shared.screenBgColor.withAlphaComponent(0.5)

        if transaction.tranType == TransactionType.walletTopup.rawValue {
            cell.logo.image = UIImage(systemName: "arrow.down.left")
            cell.logo.tintColor = DefaultAppearance.shared.primaryColor
            cell.bgLogo.backgroundColor = DefaultAppearance.shared.primaryColor.withAlphaComponent(0.1)
            cell.amountLabel.textColor = DefaultAppearance.shared.primaryColor
            cell.amountLabel.text = transaction.amountDisplay
            
        } else {
            cell.logo.image = UIImage(systemName: "arrow.up.left")
            cell.logo.tintColor = DefaultAppearance.shared.dangerColor
            cell.bgLogo.backgroundColor = DefaultAppearance.shared.dangerColor.withAlphaComponent(0.1)
            cell.amountLabel.textColor = DefaultAppearance.shared.dangerColor
            cell.amountLabel.text = "-" + transaction.amountDisplay
        }

        cell.tranDateLabel.text = B24PaymentSdkHelper.formatTransactionDate(transaction.tranDate)
//        cell.tranDateLabel.font = B24PaymentSdkHelper.setFont(named: FixFontFamily.fontEnglishMedium, ofSize: FixFontSize.tranText)
        cell.tranDateLabel.font = FontManager.shared.mediumFont(forLanguage: "en",fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.tranText)
        cell.tranDateLabel.textColor = DefaultAppearance.shared.secondaryLabelColor

        cell.detailLabel.text = (language == "km") ? (transaction.descriptionKm ) : (transaction.descriptionEn )
//        cell.detailLabel.font = B24PaymentSdkHelper.setFont(named: FixFontFamily.fontEnglishRegular, ofSize: FixFontSize.tranTypeText)
        cell.detailLabel.font = FontManager.shared.regularFont(forLanguage: "en",fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.tranTypeText)
        cell.detailLabel.textColor = DefaultAppearance.shared.primaryLabelColor.withAlphaComponent(0.4)

        
//        cell.amountLabel.font = B24PaymentSdkHelper.setFont(named: FixFontFamily.fontEnglishMedium, ofSize: FixFontSize.buttonText)
        cell.amountLabel.font = FontManager.shared.mediumFont(forLanguage: "en",fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.buttonText)
    }

    func collectionView(_ collectionView: UICollectionView, didSelectItemAt indexPath: IndexPath) {
        let selectedTransaction = transactions[indexPath.item]
        let storyboard = UIStoryboard(name: "InstantPaymentMethodView", bundle: B24PaymentSdkHelper.frameworkBundle())
        guard let transactionDetailVC = storyboard.instantiateViewController(withIdentifier: "TransactionDetailView") as? TransactionDetailView else { return }

        // Pass the transaction ID to the detail view
        transactionDetailVC.tranID = selectedTransaction.tranID

        transactionDetailVC.modalPresentationStyle = .pageSheet
        self.present(transactionDetailVC, animated: true)
    }

    // MARK: - Collection View Flow Layout Delegate
    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, sizeForItemAt indexPath: IndexPath) -> CGSize {
        let width = collectionView.frame.width
        let height: CGFloat = 80 // Fixed height for consistent alignment
        return CGSize(width: width, height: height)
    }

    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, minimumLineSpacingForSectionAt section: Int) -> CGFloat {
        return 0 // No spacing between rows for clean appearance
    }

    func collectionView(_ collectionView: UICollectionView, layout collectionViewLayout: UICollectionViewLayout, minimumInteritemSpacingForSectionAt section: Int) -> CGFloat {
        return 0 // No spacing between items in the same row
    }

}
