//
//  B24PaymentSdkHelper.swift
//  Bill24OnlinePaymentSdk
//
//  Created by MacbookPro on 12/10/23.
//
import Foundation
import UIKit

public class B24PaymentSdkHelper {
    var vc: BottomSheetViewController
    
    public static func frameworkBundle() -> Bundle? {
        var bundle: Bundle? = nil
        let bundlePath = Bundle(for: B24PaymentSdk.self).path(forResource: "B24PaymentSdk", ofType: "bundle")
        if(bundlePath != nil)
        {
            bundle = Bundle(path: bundlePath!)
        }else{
            bundle = Bundle(identifier: "com.bill24.B24PaymentSdk")
        }
        return bundle
    }
    
    public init (){
        // Load the storyboard
        let storyboard = UIStoryboard(name: "BottomSheet", bundle: B24PaymentSdkHelper.frameworkBundle())
        // Instantiate view controller
        vc = storyboard.instantiateViewController(withIdentifier: "BottomSheetViewController") as! BottomSheetViewController
    }
    
    public func presentBottomSheet(from controller: UIViewController, view: UIView, viewType: BottomSheetViewType, transactionId: String, completed: (() -> Void)? = nil) {
        self.vc.modalPresentationStyle = .overFullScreen
        self.vc.loadViewIfNeeded()
        self.vc.viewType = viewType
        self.vc.transactionId = transactionId
        self.vc.bottomSheetContainerView.addSubview(view)
        self.vc.completed = completed
        
        // Check if the view is KHQRView and notify the bottom sheet
        if view is KHQRView {
            self.vc.notifyKHQRViewAdded()
        }
        
        // Only set onPaymentSuccess handler for TopUp transactions
        // This prevents double presentation of success screens
        vc.onPaymentSuccess = { [weak self] data in
            // Only show WalletSuccessViewController if this is a top-up transaction
            if TopUpView.isTopUpTransaction {
                // Reset the flag immediately to prevent double-presentation
                TopUpView.isTopUpTransaction = false
                
                let storyboard = UIStoryboard(name: "InstantPaymentMethodView", bundle: B24PaymentSdkHelper.frameworkBundle())
                let walletVC = storyboard.instantiateViewController(withIdentifier: "WalletSuccessViewController") as! WalletSuccessViewController
                walletVC.modalPresentationStyle = .fullScreen
                
                if let rootVC = UIApplication.shared.windows.first?.rootViewController {
                    var topVC = rootVC
                    while let presented = topVC.presentedViewController {
                        topVC = presented
                    }
                    topVC.present(walletVC, animated: true, completion: nil)
                }
            }
        }
        
        controller.present(vc, animated: true, completion: nil)
        return
    }
    
    public static func navigationToNextBottomSheet(from viewController: UIViewController, completion: @escaping () -> Void) {
        if let currentBottomSheet = findTopViewController(from: viewController) as? BottomSheetViewController {
            // Dismiss currently presented bottom sheet if any
            currentBottomSheet.dismiss(animated: true) {
                // Call the completion block to present the new bottom sheet
                completion()
            }
        } else {
            // If no bottom sheet is currently presented, directly present the new one
            completion()
        }
    }
    
    public static func findTopViewController(from viewController: UIViewController) -> UIViewController? {
        var topController: UIViewController? = viewController
        while let presentedViewController = topController?.presentedViewController {
            topController = presentedViewController
        }
        return topController
    }
    
    public static func errorSnackbar(view: UIView, activitiyIndicator: UIActivityIndicatorView? = nil, message: String, forBottomSheet: Bool = false){
        let snackbar = SnackbarView(frame: CGRect(x: 0, y: !forBottomSheet ? view.frame.size.height: BottomSheetViewController.bottomSheetHeight, width: view.frame.size.width, height: 80),
                                    backgroundColor:  UIColor(
                                        named: "error_snackbar_background_color",
                                        in: B24PaymentSdkHelper.frameworkBundle(),
                                        compatibleWith: nil)!)
        
        snackbar.message.text = message
        view.addSubview(snackbar)
        
        UIView.animate(withDuration: 0.3, animations: {
            snackbar.frame.origin.y -= snackbar.frame.size.height
        }) { _ in
            DispatchQueue.main.asyncAfter(deadline: .now() + 3.0) {
                UIView.animate(withDuration: 0.3, animations: {
                    snackbar.frame.origin.y += snackbar.frame.size.height
                }) { _ in
                    snackbar.removeFromSuperview()
                }
            }
        }
        activitiyIndicator?.stopAnimating()
        activitiyIndicator?.isHidden = true
    }
    
    public static func successSnackbar(view: UIView,  message: String, forBottomSheet: Bool = false){
//        let snackbar = SnackbarView(frame: CGRect(x: 0, y: !forBottomSheet ? view.frame.size.height : BottomSheetViewController.bottomSheetHeight, width: view.frame.size.width, height: 80), backgroundColor:  UIColor(
//            named: "success_snackbar_background_color",
//            in: B24PaymentSdkHelper.frameworkBundle(),
//            compatibleWith: nil)!
//        )
        let backgroundColor = Themes.Property.alertBackgroundColor.isEmpty ? "#EBFCF6" : Themes.Property.alertBackgroundColor
        let textColor = Themes.Property.alertTextColor.isEmpty ? "#171717" : Themes.Property.alertTextColor

        let snackbar = SnackbarView(frame: CGRect(x: 0, y: !forBottomSheet ? view.frame.size.height : BottomSheetViewController.bottomSheetHeight, width: view.frame.size.width, height: 80), backgroundColor:  UIColor(hex: backgroundColor)
        )
        
        snackbar.message.text = message
        snackbar.message.textColor = UIColor(hex: textColor)
        snackbar.iconImage.image = UIImage(named:"check-circle-48", in: B24PaymentSdkHelper.frameworkBundle(),
                                           compatibleWith: nil )
        view.addSubview(snackbar)
        UIView.animate(withDuration: 0.3, animations: {
            snackbar.frame.origin.y -= snackbar.frame.size.height
        }) { _ in
            DispatchQueue.main.asyncAfter(deadline: .now() + 3.0) {
                UIView.animate(withDuration: 0.3, animations: {
                    snackbar.frame.origin.y += snackbar.frame.size.height
                }) { _ in
                    snackbar.removeFromSuperview()
                }
            }
        }
    }
    
    
    
    public static func openDeeplink(deepLinkUrl: String, view: UIView, forBottomSheet: Bool = true){
        guard let deepLinkURL = URL(string: deepLinkUrl) else {
            B24PaymentSdkHelper.errorSnackbar(
                view: view,
                message: "Invalid deep link URL",
                forBottomSheet: forBottomSheet
            )
            return
        }
        if UIApplication.shared.canOpenURL(deepLinkURL) {
            UIApplication.shared.open(deepLinkURL, options: [:], completionHandler: { (success) in
                if success {
                    print("Deep link opened successfully")
                } else {
                    B24PaymentSdkHelper.errorSnackbar(
                        view: view,
                        message: "Error opening deep link",
                        forBottomSheet: forBottomSheet
                    )
                }
            })
        } else {
            B24PaymentSdkHelper.errorSnackbar(
                view: view,
                message: "Cannot open the deep link URL",
                forBottomSheet: forBottomSheet
            )
        }
        
    }
    
    public static func formateStringToDate(_ dateString: String, format: String) -> Date? {
        let dateFormatter = DateFormatter()
        dateFormatter.dateFormat = format
        return dateFormatter.date(from: dateString)
    }
    
    public static func formatDateToString(_ date: Date, format: String) -> String {
        let dateFormatter = DateFormatter()
        dateFormatter.dateFormat = format
        return dateFormatter.string(from: date)
    }
    
    /// Formats transaction date from backend format to display format
    /// - Parameter dateString: Date string from backend in format "Sep 30, 2025 10:48 AM"
    /// - Returns: Formatted date string in format "dd/MM/yyyy 10:48 AM" or original string if parsing fails
    public static func formatTransactionDate(_ dateString: String) -> String {
        // Input format from backend: "Sep 30, 2025 10:48 AM"
        let inputFormatter = DateFormatter()
        inputFormatter.dateFormat = "MMM d, yyyy h:mm a"
        inputFormatter.locale = Locale(identifier: "en_US")
        
        // Output format: "dd/MM/yyyy 10:48 AM"
        let outputFormatter = DateFormatter()
        outputFormatter.dateFormat = "dd/MM/yyyy h:mm a"
        
        guard let date = inputFormatter.date(from: dateString) else {
            // If parsing fails, return the original string
            return dateString
        }
        
        return outputFormatter.string(from: date)
    }
    
    public static func setFontKhqr(named fontName: String, ofSize fontSize: CGFloat) -> UIFont? {
        if let frameworkBundle = B24PaymentSdkHelper.frameworkBundle() {
            if let fontURL = frameworkBundle.url(forResource: fontName, withExtension: "ttf") {
                if let fontDataProvider = CGDataProvider(url: fontURL as CFURL) {
                    if let font = CGFont(fontDataProvider) {
                        var error: Unmanaged<CFError>?
                        if CTFontManagerRegisterGraphicsFont(font, &error) {
                            print("Font registered successfully.")
                        } else {
                            if let error = error?.takeRetainedValue() {
                                print("Error registering font: \(error)")
                            }
                        }
                        return UIFont(name: font.fullName! as String, size: fontSize)
                    }
                }
            }
        }
        return nil
    }
    
    public static func setFont(named fontName: String, ofSize fontSize: CGFloat) -> UIFont? {
        if let frameworkBundle = B24PaymentSdkHelper.frameworkBundle() {
            if let fontURL = frameworkBundle.url(forResource: fontName, withExtension: "ttf") {
                if let fontDataProvider = CGDataProvider(url: fontURL as CFURL) {
                    if let font = CGFont(fontDataProvider) {
//                        CTFontManagerRegisterGraphicsFont(font, nil)
                        var error: Unmanaged<CFError>?
                        if CTFontManagerRegisterGraphicsFont(font, &error) {
                            print("Font registered successfully.")
                        } else {
                            if let error = error?.takeRetainedValue() {
                                print("Error registering font: \(error)")
                            }
                        }
//                        return UIFont(name: fontName, size: fontSize)
                        return UIFont(name: fontName, size: fontSize)
                    }
                }
            }
        }
        return nil
    }
    
    public static func getCurrentLanguage(language: String?){
        if(SDKVariableSetting.currentLanguage.isEmpty){
            SDKVariableSetting.currentLanguage = language?.lowercased() ?? "km"
        }
    }
    
    public static func localized(_ key: String) -> String {
        let bundle = B24PaymentSdkHelper.frameworkBundle()
        var languageBundle: Bundle?
        
        // Check if the framework bundle and the path for the language resource exist
        if let bundle = bundle, let path = bundle.path(forResource: (SDKVariableSetting.currentLanguage.isEmpty) ? "km" : SDKVariableSetting.currentLanguage.lowercased(), ofType: "lproj") {
            languageBundle = Bundle(path: path)
        }
        // If languageBundle is nil, try to use the default "km.lproj"
        if languageBundle == nil, let defaultPath = bundle?.path(forResource: "km", ofType: "lproj") {
            languageBundle = Bundle(path: defaultPath)
        }
        
        return languageBundle?.localizedString(forKey: key, value: nil, table: nil) ?? key
    }
    
}
