//
//  BottomSheetViewController.swift
//  Bill24OnlinePaymentSdk
//
//  Created by MacbookPro on 12/10/23.
//

import Foundation
import UIKit
import SocketIO
import Alamofire

public class BottomSheetViewController: UIViewController, UIAdaptivePresentationControllerDelegate{
    @IBOutlet weak var topBarView: UIView!
    @IBOutlet weak var bottomSheetContainerView: UIView!
    private var bottomSheetHeightConstraint: NSLayoutConstraint?
    private var manager: SocketManager!
    private var socket: SocketIOClient!
    private var panGesture: UIPanGestureRecognizer!
    private var isLoading: Bool = false
    
    public var completed: (() -> Void)? = nil
    var activityViewController: UIActivityViewController?
    public var shareActivityItems: [Any] = []
    
    var viewType: BottomSheetViewType = .full
    var transactionId: String = ""
    private var alreadyCalledVerify: Bool = false
    private var isSuccessViewPresented: Bool = false
    
    // KHQR-specific properties
    private var containsKHQRView: Bool = false
    private var originalViewType: BottomSheetViewType = .sixtyPercent
    private var expandButton: UIButton?
    
    public static var bottomSheetHeight: CGFloat = 0
    
    private var controller: BottomSheetViewController!
    public var onPaymentSuccess: ((Any?) -> Void)?
    public func appearanceSetup(){
        // if !Themes.Property.indicatorColor.isEmpty {
        //     topBarView.backgroundColor = UIColor(hex: Themes.Property.indicatorColor)
        // }
        topBarView.backgroundColor = SDKVariableSetting.isDarkMode == true ? UIColor(hex:SDKVariableSetting.indicatorDark) :UIColor(hex:SDKVariableSetting.indicatorLight)
        
        if !Themes.Property.secondaryBackgroundColor.isEmpty{
            view.backgroundColor = UIColor(hex: Themes.Property.secondaryBackgroundColor)
        }
    }
    
    public override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        // Re-detect KHQR view in case it was added after viewDidLoad
        detectKHQRView()
    }
    
    public override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        if isBeingDismissed && completed != nil{
            completed!()
        }
        // Reset the success view flag when this view controller is being dismissed
        if isBeingDismissed {
            isSuccessViewPresented = false
            
            // Cleanup socket connection
            socket?.disconnect()
            socket?.removeAllHandlers()
        }
        // Re-detect KHQR view in case it was added after viewDidLoad
        detectKHQRView()
        
        NotificationCenter.default.removeObserver(self,
                                                  name: UIApplication.didBecomeActiveNotification,
                                                  object: nil)
    }
    
    public override func viewDidLoad() {
        super.viewDidLoad()
        print("**** BottomSheet ****")
        
        // Disable iOS built-in drag-to-dismiss for pageSheet
        if #available(iOS 13.0, *) {
            isModalInPresentation = false // We'll control this manually
        }
        
        // Set presentation controller delegate
        if #available(iOS 13.0, *) {
            presentationController?.delegate = self
        }
        
        appearanceSetup()
        registerBottomSheetStyleAndConstraint()
        initSocketServer(roomName: "\(APIManager.serviceName())-\(transactionId)")
        subscribeBroadcastEvent()
        panGesture = UIPanGestureRecognizer(target: self, action: #selector(handlePanGesture(_:)))
        panGesture.cancelsTouchesInView = false
        view.addGestureRecognizer(panGesture)
        
        // Check if this bottom sheet contains KHQRView
        detectKHQRView()
        
        // Register to receive the notification when back from other app
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(applicationDidBecomeActive),
                                               name: UIApplication.didBecomeActiveNotification,
                                               object: nil)
    }
    
    deinit {
        // Unregister the observer when the object is deallocated
        NotificationCenter.default.removeObserver(self,
                                                  name: UIApplication.didBecomeActiveNotification,
                                                  object: nil)
    }
    
    // Selector method that will be called when the notification is received
    @objc func applicationDidBecomeActive() {
        // Perform tasks when the application becomes active
        // Only verify if we haven't already shown success and haven't already verified
        if !isSuccessViewPresented && !alreadyCalledVerify {
            self.verifyTransaction(transactionId: transactionId, alreadyCalledVerify: self.alreadyCalledVerify)
        }
    }
    
    @objc func handlePanGesture(_ recognizer: UIPanGestureRecognizer) {
        // Prevent drag-to-close when loading
        if isLoading {
            return
        }
        
        let translation = recognizer.translation(in: view)
        let velocity = recognizer.velocity(in: view)

        switch recognizer.state {
        case .changed:
            // Only allow downward movement
            if translation.y > 0 {
                view.transform = CGAffineTransform(translationX: 0, y: translation.y)
            }
        case .ended:
            // Calculate the dismiss threshold (20% of screen height)
            let screenHeight = UIScreen.main.bounds.height
            let dismissThreshold = screenHeight * 0.2
            
            // If contains KHQR view, collapse instead of dismiss
            if containsKHQRView && translation.y >= dismissThreshold {
                collapseToMinimized()
            } else if !containsKHQRView && translation.y >= dismissThreshold {
                // Normal dismiss behavior for non-KHQR views
                dismissBottomSheet()
            } else {
                // Reset the position of the bottom sheet
                UIView.animate(withDuration: 0.3) {
                    self.view.transform = .identity
                }
            }
        default:
            break
        }
    }
    
    // MARK: - Pan Gesture Control Methods
    public func disableDragToClose() {
        panGesture?.isEnabled = false
        isLoading = true
        // Disable iOS built-in dismissal
        if #available(iOS 13.0, *) {
            isModalInPresentation = true
        }
    }
    
    public func enableDragToClose() {
        panGesture?.isEnabled = true
        isLoading = false
        // Re-enable iOS built-in dismissal
        if #available(iOS 13.0, *) {
            isModalInPresentation = false
        }
    }
    
    // Alternative methods for loading state control
    public func setLoadingState(_ loading: Bool) {
        isLoading = loading
        if #available(iOS 13.0, *) {
            isModalInPresentation = loading
        }
    }
    
    func dismissBottomSheet() {
        UIView.animate(withDuration: 0.3, animations: {
            // Move the bottom sheet off the screen for dismissal effect
            self.view.transform = CGAffineTransform(translationX: 0, y: self.view.frame.height)
        }) { _ in
            // Remove the bottom sheet from the superview or perform any desired action
            self.dismiss(animated: true)
            
        }
    }
    
    public func registerBottomSheetStyleAndConstraint(){
        topBarView.layer.cornerRadius = 4
        // Set up height constraint for bottomSheetContainerView
        bottomSheetHeightConstraint = bottomSheetContainerView.heightAnchor.constraint(equalToConstant: UIScreen.main.bounds.height / 2)
        bottomSheetHeightConstraint?.isActive = true
        
        // Set corner radius
        bottomSheetContainerView.layer.cornerRadius = 12
        
        // Call updateViewConstraints to make sure it's applied immediately
        view.setNeedsUpdateConstraints()
    }
    
    public override func updateViewConstraints() {
        if  bottomSheetHeightConstraint != nil {
            switch viewType {
            case .full:
                updateViewConstraintsForFull()
            case .sixtyPercent:
                updateViewConstraintsForSixtyPercent()
            case .thirtyPercent:
                updateViewConstraintsForThirtyPercent()
            case .collapsed:
                updateViewConstraintsForCollapsed()
            }
        }
        super.updateViewConstraints()
    }
    
    private func updateViewConstraintsForFull() {
        self.view.layer.cornerRadius = 12
        self.view.frame.size.height = UIScreen.main.bounds.height
        self.view.frame.origin.y = 0
        BottomSheetViewController.bottomSheetHeight = self.view.frame.size.height + self.view.frame.origin.y - 80
        
        // Show the container content when not collapsed
        bottomSheetContainerView.isHidden = false
        topBarView.isHidden = false
    }
    
    private func updateViewConstraintsForSixtyPercent() {
        // Implement layout for 65% height here
        self.view.layer.cornerRadius = 12
        self.view.frame.size.height = UIScreen.main.bounds.height * 0.65
        self.view.frame.origin.y = UIScreen.main.bounds.height * 0.175
        BottomSheetViewController.bottomSheetHeight = self.view.frame.size.height + self.view.frame.origin.y - 80
        
        // Show the container content when not collapsed
        bottomSheetContainerView.isHidden = false
        topBarView.isHidden = false
    }
    
    private func updateViewConstraintsForThirtyPercent() {
        // Implement layout for ExpireView with safe area consideration
        self.view.layer.cornerRadius = 12
        
        // Get safe area bottom, fallback to a reasonable default for devices with home indicator
        let safeAreaBottom = max(view.safeAreaInsets.bottom, 34.0) // 34pt is typical for devices with home indicator
        let targetHeight = UIScreen.main.bounds.height * 0.35
        
        self.view.frame.size.height = targetHeight
        // Position above the safe area with extra padding (30 more points)
        self.view.frame.origin.y = UIScreen.main.bounds.height - targetHeight - safeAreaBottom - 40
        BottomSheetViewController.bottomSheetHeight = self.view.frame.size.height - 80
        
        // Show the container content when not collapsed
        bottomSheetContainerView.isHidden = false
        topBarView.isHidden = false
    }
    
    private func updateViewConstraintsForCollapsed() {
        // Implement layout for collapsed state (18% of screen height)
        self.view.layer.cornerRadius = 12
        self.view.frame.size.height = UIScreen.main.bounds.height * 0.06
        self.view.frame.origin.y = UIScreen.main.bounds.height - self.view.frame.size.height - (view.safeAreaInsets.bottom)
        BottomSheetViewController.bottomSheetHeight = self.view.frame.size.height
        
        // Hide the container content when collapsed
        bottomSheetContainerView.isHidden = true
        topBarView.isHidden = true
    }
    
    // MARK: - KHQR Specific Methods
    private func detectKHQRView() {
        // Check if any subview in bottomSheetContainerView is of type KHQRView
        func checkForKHQRView(in view: UIView) -> Bool {
            if view is KHQRView {
                return true
            }
            for subview in view.subviews {
                if checkForKHQRView(in: subview) {
                    return true
                }
            }
            return false
        }
        
        containsKHQRView = checkForKHQRView(in: bottomSheetContainerView)
        if containsKHQRView {
            originalViewType = viewType
        }
    }
    
    // Public method to disable KHQR behavior (for ExpireView)
    public func disableKHQRBehavior() {
        containsKHQRView = false
        
        // Re-enable dismissal
        if #available(iOS 13.0, *) {
            isModalInPresentation = false
        }
        
        // Remove expand button if it exists
        expandButton?.removeFromSuperview()
        expandButton = nil
    }
    
    public func collapseToMinimized() {
        // Store original view type if not already stored
        if originalViewType == .collapsed {
            originalViewType = viewType
        }
        
        viewType = .collapsed
        
        UIView.animate(withDuration: 0.3, animations: {
            self.view.transform = .identity
            self.updateViewConstraints()
            self.view.layoutIfNeeded()
        }) { _ in
            self.setupExpandButton()
        }
    }
    
    private func setupExpandButton() {
        // Remove existing expand button if any
        expandButton?.removeFromSuperview()
        
        // Create expand button
        expandButton = UIButton(type: .system)
        guard let button = expandButton else { return }
        
        // Use SF Symbols expand icon instead of text
        if #available(iOS 13.0, *) {
            let config = UIImage.SymbolConfiguration(pointSize: 14, weight: .bold, scale: .medium)
            let expandIcon = UIImage(systemName: "chevron.up", withConfiguration: config)
            button.setImage(expandIcon, for: .normal)
            button.imageView?.contentMode = .scaleAspectFit
        } else {
            // Fallback for older iOS versions
            button.setTitle("⬆", for: .normal)
            button.titleLabel?.font = UIFont.systemFont(ofSize: 10, weight: .bold)
        }
        
        button.backgroundColor = UIColor(hex: Themes.Property.buttonBackgroundColor)
        button.tintColor = UIColor(hex: Themes.Property.buttonTextColor)
        button.layer.cornerRadius = 15
        button.addTarget(self, action: #selector(expandButtonTapped), for: .touchUpInside)
        
        view.addSubview(button)
        button.translatesAutoresizingMaskIntoConstraints = false
        NSLayoutConstraint.activate([
            button.centerXAnchor.constraint(equalTo: view.safeAreaLayoutGuide.centerXAnchor),
            button.topAnchor.constraint(equalTo: view.safeAreaLayoutGuide.topAnchor, constant: 4),
            button.widthAnchor.constraint(equalToConstant: 30),
            button.heightAnchor.constraint(equalToConstant: 30)
        ])
    }
    
    @objc private func expandButtonTapped() {
        expandToOriginalSize()
    }
    
    private func expandToOriginalSize() {
        // Remove expand button
        expandButton?.removeFromSuperview()
        expandButton = nil
        
        // Restore original view type
        viewType = originalViewType
        
        UIView.animate(withDuration: 0.3, animations: {
            self.updateViewConstraints()
            self.view.layoutIfNeeded()
        })
    }
    
    // Public method to be called when KHQRView is added
    public func notifyKHQRViewAdded() {
        containsKHQRView = true
        originalViewType = viewType
        
        // Disable dismissal
        if #available(iOS 13.0, *) {
            isModalInPresentation = true
        }
    }
    
    private func initSocketServer(roomName: String){
        let config: SocketIOClientConfiguration = [.log(true), .compress, .forceWebsockets(true)]
        manager = SocketManager(socketURL: APIManager.socketServerUrl(), config: config)
        socket = manager.defaultSocket
        
        socket.on(clientEvent: .connect) { [self] data, ack in
            print("**** Socket Connected *****")
            socket.emit("joinRoom", "\(APIManager.serviceName())-\(transactionId)");
        }
        
        socket.on(clientEvent: .disconnect) { data, ack in
            print("*** Socket disconnected ***")
            
        }
        socket.connect()
    }
    
   var topupValue:String = ""
    
   private func getSharePref(){
        if let topupKey = SharedPreferenceManager.getString(forKey: SharePrefKey.topup.rawValue){
            self.topupValue = topupKey
        }else{
            self.topupValue = ""
        }
    }
     
     
    
    
    private func verifyTransaction(transactionId: String, alreadyCalledVerify: Bool){
        if(alreadyCalledVerify == false && !isSuccessViewPresented){
            AF.request(CheckoutRouter.checkoutDetail(transactionId: transactionId)).validate().responseData{
                (response) in
                switch response.result{
                case .success(let data):
                    let transaction = try? JSONDecoder().decode(Transaction.self, from: data)
                    if(transaction?.code == "SUCCESS"){
                        if(transaction?.data?.transInfo.status == "success"){
                            
                            // Prevent multiple triggers - check if already processing
                            if self.isSuccessViewPresented {
                                return
                            }
                            
                            // Set flags immediately to prevent race conditions
                            self.isSuccessViewPresented = true
                            self.alreadyCalledVerify = true
                            
                            // Disconnect socket FIRST to prevent further broadcasts
                            self.socket?.disconnect()
                            self.socket?.removeAllHandlers()
                            
                            let displayDefaultSuccessPage = transaction?.data?.checkoutPageConfig.displaySuccessPage
                            let merchantDeeplink = transaction?.data?.transInfo.redirectURL
                            
                            // Check TopUpView's static flag instead of SharedPreference
                            //open success of wallet
                            
                            if (TopUpView.isTopUpTransaction && displayDefaultSuccessPage == true){
                               
                                DispatchQueue.main.async {
//                                  guard let transaction = transaction else { return }
                                    // Call success handler on parent (safe VC)
                                    self.dismiss(animated: true) {
                                        // Only AFTER dismissal completes → trigger success
                                        self.onPaymentSuccess?(transaction)
                                    }
                                }
                                
                                // Exit early to prevent any further execution
                                return
                            }
                            
                            //open success deeplink
                           else if(displayDefaultSuccessPage == true){
                                
                                DispatchQueue.main.async {
                                    // Check one more time before presenting
                                    guard !self.isBeingDismissed && self.presentedViewController == nil else { return }
                                    
                                    //check to open success in wallet
                                    let storyboard = UIStoryboard(name: "SuccessScreenViewController", bundle: B24PaymentSdkHelper.frameworkBundle())
                                    let vc = storyboard.instantiateViewController(withIdentifier: "SuccessScreenViewController") as! SuccessScreenViewController
                                    vc.transactionSuccess = transaction
                                    vc.presentingVC = self
                                    vc.modalPresentationStyle = .fullScreen
                                    self.present(vc, animated: true)
                                }
                                
                                // Exit early to prevent any further execution
                                return
                               
                            }else{
                                DispatchQueue.main.async {
                                    self.dismiss(animated: true){
                                        B24PaymentSdkHelper.openDeeplink(deepLinkUrl: merchantDeeplink ?? "", view: self.view)
                                    }
                                }
                                
                                // Exit early to prevent any further execution
                                return
                            }
                        }
                    }else{
//                        print("Error\(String(describing: transaction?.messageKh))")
//                        B24PaymentSdkHelper.errorSnackbar(
//                            view: self.view,
//                            message: (transaction?.messageKh ?? transaction?.message) ??  B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
//                            forBottomSheet: false
//                        )
                        //let lan = SDKVariableSetting.currentLanguage?.lowercased() ?? "km"
                        let message = SDKVariableSetting.currentLanguage.lowercased() == "en" ? transaction?.message : transaction?.messageKh

                        B24PaymentSdkHelper.errorSnackbar(
                            view: self.view,
                            message: message ?? B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                            forBottomSheet: false
                        )
                    }
                case .failure(let error):
                    print(error)
                    B24PaymentSdkHelper.errorSnackbar(
                        view: self.view,
                        message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                        forBottomSheet: false
                    )
                }
            }
        }
    }
    
    private func subscribeBroadcastEvent(){
        socket.on("broadcast") { data, ack in
            if let dictionary = data[0] as? [String: Any],
               let transactionId = dictionary["transactionId"] as? String {
                self.verifyTransaction(transactionId: transactionId, alreadyCalledVerify: self.alreadyCalledVerify)
            }
        }
    }
}

// MARK: - UIAdaptivePresentationControllerDelegate
extension BottomSheetViewController {
    public func presentationControllerShouldDismiss(_ presentationController: UIPresentationController) -> Bool {
        // Prevent system dismissal when loading or when containing KHQR view
        return !isLoading && !containsKHQRView
    }
}







