//
//  KHQRView.swift
//  Bill24OnlinePaymentSdk
//
//  Created by MacbookPro on 23/10/23.
//

import Foundation
import UIKit
import Alamofire

public class KHQRView : UIView{
    @IBOutlet weak var khqrView: UIView!
    @IBOutlet weak var khqrContainer: UIView!
    @IBOutlet weak var khqrHeaderContainer: UIView!
    @IBOutlet weak var downloadButton: UIView!
    @IBOutlet weak var shareButton: UIView!
    @IBOutlet weak var countDownLabel: UILabel!
    @IBOutlet weak var orLabel: UILabel!
    @IBOutlet weak var activityIndicator: UIActivityIndicatorView!
    @IBOutlet weak var orViewLabel: UIView!
    @IBOutlet weak var dotLineView: DottedLineView!
    @IBOutlet weak var downloadLabel: UILabel!
    @IBOutlet weak var shareLabel: UILabel!
    
    @IBOutlet weak var shareIcon: UIImageView!
    @IBOutlet weak var downloadIcon: UIImageView!
    @IBOutlet weak var scanToPayLabel: UILabel!
    @IBOutlet weak var displayAccountBillerName: UILabel!
    @IBOutlet weak var currencyLabel: UILabel!
    @IBOutlet weak var totalAmountDisplay: UILabel!
    @IBOutlet weak var currencyIcon: UIImageView!
    @IBOutlet weak var khqrImageView: UIImageView!
    private let countdownManager = CountdownManager()
    private var expireDateResponse: ExtendExpireDateResponse?
    private var khqrMainView: UIView!
    private var  centerLoading = UIActivityIndicatorView(style: .large)
    private var stackView : UIStackView!
    private var dismissButton: UIButton!
    
    var transactionId: String = ""
    var controller: UIViewController!
    var transactionData: Transaction?
    private var bottomSheetController: BottomSheetViewController?
    private var alreadyCalledVerify = false
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        initSubviews()
    }
    
    init(frame: CGRect, transactionId: String, controller: UIViewController, transactionData: Transaction, bottomSheetController: BottomSheetViewController) {
        self.transactionId = transactionId
        self.controller = controller
        self.transactionData = transactionData
        self.bottomSheetController = bottomSheetController
        super.init(frame: frame)
        initSubviews()
    }
    
    private func initSubviews() {
        registerView()
        setUpDataToKhqrView()
        setUpConstraintAndStyle()
        setUpFont()
        setUpTheme()
        setUpTranslate()
        extendExpiredTransactionApi()
        
        downloadButton.isUserInteractionEnabled = true
        downloadButton.addGestureRecognizer(UITapGestureRecognizer(target: self, action: #selector(downloadImage)))
        
        shareButton.isUserInteractionEnabled = true
        shareButton.addGestureRecognizer(UITapGestureRecognizer(target: self, action: #selector(shareImage)))
        
        // Set up dismiss button
        setupDismissButton()
    }
    
    private func registerView(){
        let nib = UINib(
            nibName: "KHQRView",
            bundle: B24PaymentSdkHelper.frameworkBundle()
        )
        
        guard let view = nib.instantiate(withOwner: self, options: nil).first as?
                UIView else {fatalError("Unable to convert nib")}
        view.frame = bounds
        view.autoresizingMask = [.flexibleWidth, .flexibleHeight]
        khqrMainView = view
        addSubview(view)
        
        // Configure the activity indicator
        centerLoading.center =  khqrView.center
        centerLoading.hidesWhenStopped = true
        centerLoading.color = UIColor.systemBlue
        view.addSubview(centerLoading)
        
        // Config Error View for Extend Expire Endpoint
        // Create UIImageView for the image
        let imageView = UIImageView(image: UIImage(named: "expire_icon",  in: B24PaymentSdkHelper.frameworkBundle(),compatibleWith: nil))
        imageView.contentMode = .center
        imageView.widthAnchor.constraint(equalToConstant: 60).isActive = true
        imageView.heightAnchor.constraint(equalToConstant: 60).isActive = true

        // Create UILabel for the text
        let label = UILabel()
        label.text = B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue)
        label.textAlignment = .center
        label.font =  FontManager.shared.regularFont(forLanguage: SDKVariableSetting.currentLanguage, size: 15.0)
        label.textColor = UIColor(hex: Themes.Property.secondaryTextColor)

        // Create UIStackView
        stackView = UIStackView(arrangedSubviews: [imageView, label])
        stackView.axis = .vertical
        stackView.alignment = .center
        stackView.distribution = .fillEqually

        // Add the stack view to your main view or another container view
        view.addSubview(stackView)

        // Set constraints for the stack view
        stackView.translatesAutoresizingMaskIntoConstraints = false
        NSLayoutConstraint.activate([
            stackView.centerXAnchor.constraint(equalTo: khqrView.centerXAnchor),
            stackView.centerYAnchor.constraint(equalTo: khqrView.centerYAnchor),
        ])
        
        view.layoutIfNeeded()
       
        self.showCenterLoading()
        view.backgroundColor = UIColor(hex: Themes.Property.secondaryBackgroundColor)
        
        print("*** Load KHQR View ***")
    }
    
    private func setupDismissButton() {
        // Create dismiss button
        dismissButton = UIButton(type: .system)
        dismissButton.setImage(UIImage(systemName: "xmark", withConfiguration: UIImage.SymbolConfiguration(weight: .bold)), for: .normal)
        dismissButton.tintColor = UIColor(hex: Themes.Property.primaryTextColor)
        dismissButton.backgroundColor = UIColor.clear
        dismissButton.translatesAutoresizingMaskIntoConstraints = false
        dismissButton.addTarget(self, action: #selector(dismissBottomSheet), for: .touchUpInside)
        
        // Add button to the main view
        khqrMainView.addSubview(dismissButton)
        
        // Set up constraints to position it next to the countdown label
        NSLayoutConstraint.activate([
            dismissButton.centerYAnchor.constraint(equalTo: countDownLabel.centerYAnchor),
            dismissButton.leadingAnchor.constraint(equalTo: countDownLabel.trailingAnchor, constant: 8),
            dismissButton.widthAnchor.constraint(equalToConstant: 16),
            dismissButton.heightAnchor.constraint(equalToConstant: 16)
        ])
    }
    
    @objc private func dismissBottomSheet() {
        bottomSheetController?.dismiss(animated: true)
    }
    
    private func setUpTranslate(){
        scanToPayLabel.text = B24PaymentSdkHelper.localized(KhqrLocalizedKeys.scanToPay.rawValue)
        downloadLabel.text = B24PaymentSdkHelper.localized(LocalizedKeys.download.rawValue)
        shareLabel.text = B24PaymentSdkHelper.localized(LocalizedKeys.share.rawValue)
        orLabel.text = B24PaymentSdkHelper.localized(KhqrLocalizedKeys.or.rawValue)
    }
    
    private func setUpTheme(){
        scanToPayLabel.textColor = UIColor(hex: Themes.Property.primaryTextColor)
        orLabel.textColor =  UIColor(hex: Themes.Property.primaryTextColor)
        dotLineView.tintColor = UIColor(hex: Themes.Property.indicatorColor)
        downloadLabel.textColor = UIColor(hex: Themes.Property.primaryTextColor)
        shareLabel.textColor = UIColor(hex: Themes.Property.primaryTextColor)
        downloadButton.backgroundColor = UIColor(hex: Themes.Property.actionButtonBackgroundColor)
        downloadIcon.tintColor = UIColor(hex: Themes.Property.actionButtonTextColor)
        shareIcon.tintColor = UIColor(hex: Themes.Property.actionButtonTextColor)
        shareButton.backgroundColor = UIColor(hex: Themes.Property.actionButtonBackgroundColor)
        countDownLabel.textColor = UIColor(hex: Themes.Property.primaryTextColor)
        activityIndicator.color = UIColor(hex: Themes.Property.indicatorColor)
        orViewLabel.backgroundColor = UIColor(hex: Themes.Property.secondaryBackgroundColor)
        khqrMainView.backgroundColor = UIColor(hex: Themes.Property.secondaryBackgroundColor)
        currencyLabel.textColor =  UIColor(named: "text_primary_color", in: B24PaymentSdkHelper.frameworkBundle(), compatibleWith: nil)
        displayAccountBillerName.textColor =  UIColor(named: "text_primary_color", in: B24PaymentSdkHelper.frameworkBundle(), compatibleWith: nil)
        totalAmountDisplay.textColor = UIColor(named: "text_primary_color", in: B24PaymentSdkHelper.frameworkBundle(), compatibleWith: nil)
        
        // Apply theme to dismiss button if it exists
        if dismissButton != nil {
            dismissButton.tintColor = UIColor(hex: Themes.Property.primaryTextColor)
        }
    }
    
    private func setUpFont(){
        countDownLabel.font = B24PaymentSdkHelper.setFont(named: "Roboto-Regular", ofSize: 13.0)
        scanToPayLabel.font = FontManager.shared.regularFont(forLanguage: SDKVariableSetting.currentLanguage, size: 15.0)
        orLabel.font =  FontManager.shared.regularFont(forLanguage: SDKVariableSetting.currentLanguage, size: 13.0)
        downloadLabel.font = FontManager.shared.regularFont(forLanguage: SDKVariableSetting.currentLanguage, size: 13.0)
        shareLabel.font = FontManager.shared.regularFont(forLanguage: SDKVariableSetting.currentLanguage, size: 13.0)
        
        displayAccountBillerName.font = B24PaymentSdkHelper.setFontKhqr(named: "NunitoSans_7pt-SemiBold", ofSize: 11.0)
        currencyLabel.font = B24PaymentSdkHelper.setFontKhqr(named: "NunitoSans_7pt-SemiBold", ofSize: 11.0)
        totalAmountDisplay.font = B24PaymentSdkHelper.setFontKhqr(named: "NunitoSans_7pt-ExtraBold", ofSize: 18.0)
    }
    
    private func showCenterLoading(){
        stackView.isHidden = true
        startLoading(view: self)
        khqrView.isHidden = true
        scanToPayLabel.isHidden = true
        downloadButton.isHidden = true
        shareButton.isHidden = true
        downloadLabel.isHidden = true
        shareLabel.isHidden = true
        activityIndicator.isHidden = true
        countDownLabel.isHidden = true
        orViewLabel.isHidden = true
        dotLineView.isHidden = true
        dismissButton?.isHidden = true
    }
    
    private func hideCenterLoading(){
        stopLoading(view: self)
        khqrView.isHidden = false
        scanToPayLabel.isHidden = false
        downloadButton.isHidden = false
        shareButton.isHidden = false
        downloadLabel.isHidden = false
        shareLabel.isHidden = false
        activityIndicator.isHidden = false
        countDownLabel.isHidden = false
        orViewLabel.isHidden = false
        dotLineView.isHidden = false
        dismissButton?.isHidden = false
    }
    
    private func startLoading(view: UIView) {
        centerLoading.startAnimating()
    }
    private func stopLoading(view: UIView) {
        centerLoading.stopAnimating()
    }
    
    @objc func shareImage(){
        let image = khqrView.saveAsImage()
        let billerNameDisplay = (transactionData?.data?.biller.billerName) ?? (transactionData?.data?.biller.billerNameKh)
        let amountDisplay = (transactionData?.data?.transInfo.totalAmountDisplay) ?? ""
        let currencyDisplay = (transactionData?.data?.transInfo.currency) ?? ""
        let imageName = "\(billerNameDisplay ?? "")-\(amountDisplay) \(currencyDisplay).png"
        // Get the URL for the documents directory
        if let documentsDirectory = FileManager.default.urls(for: .documentDirectory, in: .userDomainMask).first {
            // Append the custom image name to the documents directory URL
            let fileURL = documentsDirectory.appendingPathComponent(imageName)
            
            // Save the image to the specified URL
            if let data = image.pngData() {
                do {
                    try data.write(to: fileURL)
                    let items: [Any] = [imageName,image]
                    let activityViewController = UIActivityViewController(activityItems: items, applicationActivities: nil)
                    activityViewController.popoverPresentationController?.sourceView = self
                    activityViewController.popoverPresentationController?.sourceRect = CGRect(x: self.bounds.midX, y: self.bounds.midY, width: 0, height: 0)
                    if let viewController = self.getViewController() {
                        viewController.present(activityViewController, animated: true, completion: nil)
                    }
                } catch {
                    print("Error saving image: \(error)")
                }
            }
        }
    }
    
    private func getViewController() -> UIViewController? {
        var responder: UIResponder? = self
        while let nextResponder = responder?.next {
            if let viewController = nextResponder as? UIViewController {
                return viewController
            }
            responder = nextResponder
        }
        return nil
    }
    
    @objc func downloadImage(){
        let image = khqrView.saveAsImage()
        saveImageToPhotoLibrary(image: image)
    }
    
    private func saveImageToPhotoLibrary(image: UIImage) {
        UIImageWriteToSavedPhotosAlbum(image, self, #selector(image(_:didFinishSavingWithError:contextInfo:)), nil)
    }
    
    @objc func image(_ image: UIImage, didFinishSavingWithError error: NSError?, contextInfo: UnsafeRawPointer) {
        if let error = error {
            print(error)
            B24PaymentSdkHelper.errorSnackbar(
                view: self,
                message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                forBottomSheet: true
            )
        } else {
            B24PaymentSdkHelper.successSnackbar(
                view: self,
                message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.saveKhqr.rawValue),
                forBottomSheet: true
            )
        }
    }
    
    private func setUpDataToKhqrView(){
        let data = transactionData?.data
        khqrImageView.loadQRImage(from: data?.transInfo.khqrString.data(using: .utf8))
        displayAccountBillerName.text = data?.biller.billerDisplayName
        totalAmountDisplay.text = data?.transInfo.totalAmountDisplay
        currencyLabel.text = data?.transInfo.currency
     
        // Display KHQR Currency base on currency
        if(data?.transInfo.currency.lowercased() == "khr"){
            currencyIcon.image = UIImage(named: "khr_currency" ,in: B24PaymentSdkHelper.frameworkBundle(), compatibleWith: nil)
        }else{
            currencyIcon.image = UIImage(named: "usd_currency" ,in: B24PaymentSdkHelper.frameworkBundle(), compatibleWith: nil)
        }
        
       
    }
    
    private func extendExpiredTransactionApi(){
        AF.request(CheckoutRouter.updateTransactionExpire(transactionId: transactionId)).validate().responseData{
            [self] response in
            switch response.result{
            case .success(let data):
                expireDateResponse = try? JSONDecoder().decode(ExtendExpireDateResponse.self, from: data)
                if(expireDateResponse?.code == "SUCCESS"){
                    // Call requestVeriftyTransaction to verify and setup KHQR
                    self.requestVeriftyTransaction()
                }else{
                    stopLoading(view: self)
                    stackView.isHidden = false
                    
                }
            case .failure(let error):
                print(error)
                stopLoading(view: self)
                stackView.isHidden = false
            }
        }
    }
    
    // Removed SharedPreference approach - now using TopUpView.isTopUpTransaction static variable 
    private func requestVeriftyTransaction() {
        AF.request(CheckoutRouter.checkoutDetail(transactionId: transactionId)).validate().responseData { response in
            switch response.result {
            case .success(let data):
                let transaction = try? JSONDecoder().decode(Transaction.self, from: data)
                
                if transaction?.code == "SUCCESS" {                        
                    // Update transaction data with new KHQR information
                    self.transactionData = transaction
                    
                    // Update KHQR display with new data
                    self.setUpDataToKhqrView()
                    
                    // Setup KHQR with countdown
                    let secondRemainingTime = self.calculateSessionExpire(expireDate: (transaction?.data?.transInfo.expiredDate)!)
                    if(secondRemainingTime > 3600){
                        // second grater than 1 hour
                        self.countDownLabel.isHidden = true
                        self.activityIndicator.isHidden = true
                        self.dismissButton?.isHidden = false // Keep dismiss button visible even when countdown is hidden
                    }else{
                        self.countDownLabel.isHidden = false
                        self.activityIndicator.isHidden = false
                        self.dismissButton?.isHidden = false
                        self.updateCountdown(seconds: secondRemainingTime)
                    }
                    self.hideCenterLoading()
                } else {
                    // let lan = SDKVariableSetting.currentLanguage?.lowercased() ?? "km"
                    let message = SDKVariableSetting.currentLanguage.lowercased() == "en" ? transaction?.message : transaction?.messageKh
                    
                    self.stopLoading(view: self)
                    self.stackView.isHidden = false
                    
                    // Show snackbar on the superview (bottomSheetContainerView) to ensure visibility
                    if let parentView = self.superview {
                        B24PaymentSdkHelper.errorSnackbar(
                            view: parentView,
                            message: message ?? B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                            forBottomSheet: false
                        )
                    }
                }
                
            case .failure(let error):
                print(error)
                self.stopLoading(view: self)
                self.stackView.isHidden = false
                
                // Show snackbar on the superview (bottomSheetContainerView) to ensure visibility
                if let parentView = self.superview {
                    B24PaymentSdkHelper.errorSnackbar(
                        view: parentView,
                        message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                        forBottomSheet: false
                    )
                }
            }
        }
    }

    private func calculateSessionExpire(expireDate: String) -> Int {
        var secondRemain: Int = 0
        if let expireDate = B24PaymentSdkHelper.formateStringToDate(expireDate, format: "yyyy-MM-dd HH:mm:ss") {
            let currentDate = Date()
            secondRemain = Int(expireDate.timeIntervalSince(currentDate))
        } else {
            print("Invalid date string")
        }
        
        return secondRemain
    }
    
    private func updateCountdown(seconds: Int) {
        countdownManager.startCountdown(seconds: seconds, updateLabel: { [weak self] timeString in
            self?.countDownLabel.text = timeString
            self?.activityIndicator.startAnimating()
        }, completion: {
            self.navigateToExpire()
        })
    }
    
    func navigateToExpire(){
        if(countdownManager.secondsRemaining == 0){
            self.removeFromSuperview()
            let expireView = ExpireView(
                frame: controller.view.frame,
                transactionId: transactionId,
                controller: controller,
                transactionData: transactionData!
            )
            bottomSheetController!.bottomSheetContainerView.addSubview(expireView)
            bottomSheetController!.viewType = .thirtyPercent
            bottomSheetController!.updateViewConstraints()
            
            // Disable KHQR behavior for ExpireView (make it dismissible)
            bottomSheetController!.disableKHQRBehavior()
        }
    }
    
    public override func removeFromSuperview() {
        super.removeFromSuperview()
        countdownManager.stopCountdown()
    }
    
    public func completed() {
        if(countdownManager.secondsRemaining > 0){
            countdownManager.stopCountdown()
        }
    }
    
    private func setUpConstraintAndStyle(){
        khqrContainer.layer.cornerRadius = 30
        let shadowColor =  UIColor(red: 0, green: 0, blue: 0)
        khqrContainer.layer.shadowColor = shadowColor.cgColor
        khqrContainer.layer.shadowOffset = CGSize(width: 0, height: 0)
        khqrContainer.layer.shadowOpacity = 0.10
        khqrContainer.layer.shadowRadius = 8
        khqrContainer.layer.masksToBounds = false
        
        khqrHeaderContainer.roundCorners(corners: [.topLeft, .topRight], radius: 30.0)
        downloadButton.layer.cornerRadius = 10
        shareButton.layer.cornerRadius = 10
        
        downloadButton.backgroundColor =  UIColor(hex: "#0476FB2B")
        shareButton.backgroundColor = UIColor(hex: "#0476FB2B")
    }
}



