//
//  AccountDetailView.swift
//  B24PaymentSdk
//
//  Created by visal ny on 22/12/24.
//

import UIKit
import Alamofire

class AccountDetailView: UIViewController, UIGestureRecognizerDelegate, TransactionsViewDelegate {
    
    
    @IBOutlet weak var transactionCollectionView: UICollectionView!
    @IBOutlet weak var containerHeader: UIView!
    @IBOutlet weak var bgLogo: UIView!
    @IBOutlet weak var lblTitle: UILabel!
    @IBOutlet weak var iconEdit: UIImageView!
    @IBOutlet weak var logo: UIImageView!
    @IBOutlet weak var contentContainer: UIView!
    @IBOutlet weak var accountNoText: UILabel!
    @IBOutlet weak var accountNameText: UILabel!
    @IBOutlet weak var currencyText: UILabel!
    @IBOutlet weak var accountNoValue: UILabel!
    @IBOutlet weak var accountNameValue: UILabel!
    @IBOutlet weak var currencyValue: UILabel!
    @IBOutlet weak var setDefaultButtonContainer: UIView!
    @IBOutlet weak var setDefaultIcon: UIImageView!
    @IBOutlet weak var setDefaultLabel: UILabel!
    @IBOutlet weak var disableContainer: UIView!
    @IBOutlet weak var disableIcon: UIImageView!
    @IBOutlet weak var disableLabel: UILabel!
    @IBOutlet weak var topupButton: UIButton!
    
    @IBOutlet weak var inactiveLabel: UILabel!
    @IBOutlet weak var bgInactive: UIView!
    @IBOutlet weak var accNoDotView: UILabel!
    
    @IBOutlet weak var accNameDotView: UILabel!
    
    @IBOutlet weak var progressIndicatorMain: UIActivityIndicatorView!
    @IBOutlet weak var progressContainer: UIView!
    @IBOutlet weak var currencyDotView: UILabel!
    
    @IBOutlet weak var emptyLabel: UILabel!

    @IBOutlet weak var toolbarLine: UIView!
    @IBOutlet weak var emptyImage: UIImageView!
    var dataAccountDetail = DataAccountDetail()
    var accountId: String = ""
    var paymentMethodId:String = ""
    var language:String = "km"
    
    var isDefault:Bool = false
    var isInactive:Bool = false
    var isWalletTopup:Bool = false
    
    var accountNo:String = "..."
    var paymentMethodCount:Int = 1
    
    // Constraint for topup button spacing adjustment
    private var topupButtonTopConstraint: NSLayoutConstraint?
    
    // Configuration property to control outside tap dismissal
    var shouldDismissOnOutsideTap: Bool = false
    
    let emptyIconName = "exclamationmark.circle.fill"
    
    var minHeight:Double = 0.0

    // MARK: - Network Monitoring
    private var isWaitingForConnection: Bool = false
  
    override func viewDidLoad() {
        super.viewDidLoad()
        
        view.backgroundColor = DefaultAppearance.shared.screenBgColor
        toolbarLine.isHidden = true
        
        
        getSharePref()
        
        applyToolBar()
        
        B24PaymentSdkHelper.getCurrentLanguage(language: language)
       
        accountDetail()
        setupModifyName()
       
    }

    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        setupNetworkMonitoring()
        NetworkMonitor.shared.startMonitoring()
    }

    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        NetworkMonitor.shared.stopMonitoring()
        NetworkMonitor.shared.onConnectionStatusChanged = nil
    }

    // MARK: - Network Monitoring Setup
    private func setupNetworkMonitoring() {
        NetworkMonitor.shared.onConnectionStatusChanged = { [weak self] isConnected in
            guard let self = self else { return }
            
            if isConnected {
                guard self.isWaitingForConnection else { return }

                self.isWaitingForConnection = false

                B24PaymentSdkHelper.hideNoInternetPopup(from: self)
                // Auto-refresh account details when connection is restored
                self.accountDetail()
            }
        }
    }
    
     
    @IBAction func topUpButtonAction(_ sender: Any) {
        guard NetworkMonitor.shared.isConnected else {
            if !isWaitingForConnection {
                isWaitingForConnection = true
                setupNetworkMonitoring()
                NetworkMonitor.shared.startMonitoring()
                
                B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                    guard let self = self else { return }
                    self.isWaitingForConnection = false
                    B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                }
            }
            return
        }

        if isInactive{
            let dialog = WarningAlertView()
            dialog.showWarningTopupAlert(from: self) {
             self.updateAccountStatus(inactive: false)
            } cancelHandler: {

            }

        }else{
            let storyboard = UIStoryboard(name: "InstantPaymentMethodView", bundle: B24PaymentSdkHelper.frameworkBundle())
            if let topUpVC = storyboard.instantiateViewController(withIdentifier: "TopUpView") as? TopUpView {
                // Pass the currency from the current account detail
                topUpVC.currency = self.dataAccountDetail.currency
                topUpVC.walletId = self.dataAccountDetail.id
                topUpVC.walletName = self.dataAccountDetail.accountName
                topUpVC.wallet = self.dataAccountDetail.accountNo
                topUpVC.balance = self.dataAccountDetail.amountDisplay
                topUpVC.currencyCode=self.dataAccountDetail.currencyCode
                topUpVC.currencySymbol=self.dataAccountDetail.currencySymbol

                navigationController?.pushViewController(topUpVC, animated: true)
            }
            
        }
    }
    
    private func applyToolBar(){
        // Configure navigation bar appearance for consistency
        if #available(iOS 13.0, *) {
            let appearance = UINavigationBarAppearance()
            appearance.configureWithOpaqueBackground()
            appearance.backgroundColor = DefaultAppearance.shared.screenBgColor
            appearance.shadowColor = DefaultAppearance.shared.primaryLabelColor.withAlphaComponent(0.3)
            
            navigationController?.navigationBar.standardAppearance = appearance
            navigationController?.navigationBar.scrollEdgeAppearance = appearance
            navigationController?.navigationBar.compactAppearance = appearance
        } else {
            navigationController?.navigationBar.barTintColor = DefaultAppearance.shared.screenBgColor
            navigationController?.navigationBar.isTranslucent = false
        }
        
        let customButton = UIButton(type: .system)
        customButton.translatesAutoresizingMaskIntoConstraints = false
        // Image
        let imageView = UIImageView(image: UIImage(systemName: "chevron.backward"))
        imageView.tintColor = DefaultAppearance.shared.primaryLabelColor
        imageView.contentMode = .scaleAspectFit
        imageView.translatesAutoresizingMaskIntoConstraints = false
        customButton.addSubview(imageView)
        
        let titleLabel = UILabel()
        
        let toolbarTitle:String = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.title.rawValue)
        
        titleLabel.text = toolbarTitle + accountNo
        
        titleLabel.textColor = DefaultAppearance.shared.primaryLabelColor
        titleLabel.font = FontManager.shared.mediumFont(forLanguage: language,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.toolbarTitle)
        titleLabel.translatesAutoresizingMaskIntoConstraints = false
        customButton.addSubview(titleLabel)

        // Add constraints to position the image and title properly
        NSLayoutConstraint.activate([
            // imageView.leadingAnchor.constraint(equalTo: customButton.leadingAnchor,constant: 10),
            imageView.leadingAnchor.constraint(equalTo: customButton.leadingAnchor),
            imageView.centerYAnchor.constraint(equalTo: customButton.centerYAnchor),
            //  imageView.widthAnchor.constraint(equalToConstant: 24), // Set desired width
            imageView.widthAnchor.constraint(equalToConstant: 20),
            imageView.heightAnchor.constraint(equalToConstant: 24),
                        
                        
            //            titleLabel.leadingAnchor.constraint(equalTo: imageView.trailingAnchor, constant: 8),
            titleLabel.leadingAnchor.constraint(equalTo: imageView.trailingAnchor, constant: 4),
            titleLabel.centerYAnchor.constraint(equalTo: customButton.centerYAnchor),
            //            titleLabel.trailingAnchor.constraint(equalTo: customButton.trailingAnchor,constant: -10)
            titleLabel.trailingAnchor.constraint(equalTo: customButton.trailingAnchor, constant: -5),
        ])

        // Add action to the button
        customButton.addTarget(self, action: #selector(backButtonTapped), for: .touchUpInside)

        // Create a UIBarButtonItem with the custom button
        let backButton = UIBarButtonItem(customView: customButton)
        navigationItem.leftBarButtonItem = backButton
    }
    
    // Back button action
    @objc func backButtonTapped() {
      //  navigationController?.popViewController(animated: true)
//        if let navigationController = self.navigationController, navigationController.viewControllers.count > 1 {
//                // There's a navigation controller and more than one view controller in the stack, so we can pop
//            if(isWalletTopup){
//                dismissAll()
//            }else{
//                navigationController.popViewController(animated: true)
//            }
//                
//        } else {
//                // No navigation controller or no previous view controller in the stack
//            print("Cannot pop view controller. Either no navigation controller or at root view.")
//        }
        
        if let navigationController = navigationController {
            if navigationController.viewControllers.count > 1 {
                // Pop if not the root view controller
                if(isWalletTopup){
                    dismissAll()
                }else{
                    navigationController.popViewController(animated: true)
                }
                //navigationController.popViewController(animated: true)
            } else {
                // Dismiss if this is the root view controller
                dismiss(animated: true, completion: nil)
            }
        } else {
            // Dismiss if not in a navigation controller
            dismiss(animated: true, completion: nil)
        }
        
        
    }
    
    func dismissAll(){
        var presentingVC = presentingViewController
        while let parent = presentingVC?.presentingViewController {
            presentingVC = parent
        }
        presentingVC?.dismiss(animated: true) {
            print("Dismissal completed")
        }
    }
    
    
   private func startLoading() {
       DispatchQueue.main.async {
           self.progressContainer.isHidden = false
           self.progressContainer.backgroundColor = DefaultAppearance.shared.cardColor
           self.progressIndicatorMain.hidesWhenStopped = true
           self.progressIndicatorMain.color = DefaultAppearance.shared.primaryColor
           self.progressContainer.backgroundColor = DefaultAppearance.shared.screenBgColor
           self.progressIndicatorMain.startAnimating()
       }
    }
    
    private func stopLoading() {
        DispatchQueue.main.async {
            self.progressIndicatorMain.stopAnimating()
            self.progressContainer.isHidden = true
            self.progressIndicatorMain.isHidden = true
        }
    }
    
    
    private func getSharePref(){
        if let languageCode = SharedPreferenceManager.getString(forKey: SharePrefKey.lanuageCode.rawValue){
            language = languageCode
        }else{
            language = "km"
        }
        
        if let id = SharedPreferenceManager.getString(forKey: SharePrefKey.paymentMethodId.rawValue){
            paymentMethodId = id
        }else{
            paymentMethodId = ""
        }
        
        isWalletTopup = SharedPreferenceManager.getBool(forKey: SharePrefKey.isWalletTopup.rawValue)
    }
    
    private func headerAppearance(data:DataAccountDetail){
        containerHeader.backgroundColor = DefaultAppearance.shared.screenBgColor
        bgLogo.backgroundColor = DefaultAppearance.shared.primaryColor
        bgLogo.layer.cornerRadius = 6
        logo.tintColor = DefaultAppearance.shared.onPrimaryColor
        
        lblTitle.numberOfLines = 1
        lblTitle.lineBreakMode = .byTruncatingTail
        lblTitle.translatesAutoresizingMaskIntoConstraints = false
        lblTitle.widthAnchor.constraint(lessThanOrEqualToConstant: 200).isActive = true
        
        lblTitle.text = data.accountName
        lblTitle.textColor = DefaultAppearance.shared.primaryLabelColor
        //lblTitle.font = B24PaymentSdkHelper.setFont(named: "Roboto-Medium", ofSize: FixFontSize.toolbarTitle)
        lblTitle.font = FontManager.shared.mediumFont(forLanguage: language,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.toolbarTitle)
        
        
        iconEdit.tintColor = DefaultAppearance.shared.primaryLabelColor
        
        if isInactive{
            bgInactive.isHidden = false
            bgInactive.backgroundColor = DefaultAppearance.shared.dangerColor
            bgInactive.layer.cornerRadius = 6
            inactiveLabel.textColor = DefaultAppearance.shared.onPrimaryColor
            inactiveLabel.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: 12)
            inactiveLabel.text = B24PaymentSdkHelper.localized(PaymentMethodWalletLocalizedKeys.inActive.rawValue)
            
        }else{
            if isDefault{
                bgInactive.isHidden = false
                bgInactive.backgroundColor = DefaultAppearance.shared.primaryColor
                bgInactive.layer.cornerRadius = 6
                inactiveLabel.textColor = DefaultAppearance.shared.onPrimaryColor
                inactiveLabel.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: 12)
                inactiveLabel.text = B24PaymentSdkHelper.localized(PaymentMethodWalletLocalizedKeys.DefaultACC.rawValue)
            }else{
                bgInactive.isHidden = true
            }
        }
        
    }
    
    private func contentAppearance(data:DataAccountDetail){
        contentContainer.backgroundColor = DefaultAppearance.shared.cardColor
        contentContainer.addCardShadow(cornerRadius: 8)
        contentContainer.roundCorners(cornerRadius: 8)
        
        accNoDotView.textColor = DefaultAppearance.shared.primaryLabelColor
        accNameDotView.textColor = DefaultAppearance.shared.primaryLabelColor
        currencyDotView.textColor = DefaultAppearance.shared.primaryLabelColor
    
        
        // apply color
        accountNoText.textColor = DefaultAppearance.shared.primaryLabelColor
        accountNameText.textColor = DefaultAppearance.shared.primaryLabelColor
        currencyText.textColor = DefaultAppearance.shared.primaryLabelColor
        
        accountNoValue.textColor = DefaultAppearance.shared.primaryLabelColor
        accountNameValue.textColor = DefaultAppearance.shared.primaryLabelColor
        currencyValue.textColor = DefaultAppearance.shared.primaryLabelColor
        
        // apply text
        if data.type == AvailablePMType.wallet.rawValue{
            accountNoText.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.walletId2.rawValue)
            accountNameText.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_wallet_ballance.rawValue)
            accountNameValue.text = data.amountDisplay
        }else{
            accountNoText.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.accNo.rawValue)
            accountNameText.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_account_name.rawValue)
            accountNameValue.text = data.accountName
        }
        
        currencyText.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_currency.rawValue)
        accountNoValue.text = data.accountNo
        currencyValue.text = data.currency
       
        //apply font
        accountNoText.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.contentText)
        accountNoValue.font = FontManager.shared.regularFont(forLanguage: language , fontFromconfig: DefaultAppearance.shared.fontEnglish,size: FixFontSize.contentText)
        accountNameText.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.contentText)
        accountNameValue.font = FontManager.shared.mediumFont(forLanguage: language , fontFromconfig: DefaultAppearance.shared.fontEnglish,size: FixFontSize.contentText)
        currencyText.font = FontManager.shared.regularFont(forLanguage: language , fontFromconfig: DefaultAppearance.shared.fontEnglish,size: FixFontSize.contentText)
        currencyValue.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.contentText)
        
    }
    
    func buttonAppearance(data:DataAccountDetail){
        
        // Hide buttons and adjust spacing if paymentMethodCount == 1
        if paymentMethodCount == 1 {
            setDefaultButtonContainer.isHidden = true
            disableContainer.isHidden = true
            
            // Remove existing constraints and set new constraint for topupButton
            adjustTopupButtonSpacing()
            return
        }
        
        setDefaultButtonContainer.isHidden = false
        disableContainer.isHidden = false
        
        // Reset topup button spacing for normal case
        resetTopupButtonSpacing()
        
        setDefaultButtonContainer.backgroundColor = DefaultAppearance.shared.cardColor
        setDefaultButtonContainer.layer.cornerRadius = 8
        
        disableContainer.backgroundColor = DefaultAppearance.shared.cardColor
        disableContainer.layer.cornerRadius = 8
        
        //size
        setDefaultLabel.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_set_default.rawValue)
        setDefaultLabel.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: language == "km" ? 14 : FixFontSize.buttonText)
        disableLabel.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.buttonText)

        if isDefault{
            setDefaultIcon.tintColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
            setDefaultLabel.textColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
            disableIcon.tintColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
            disableLabel.textColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
//            disableIcon.image = UIImage(systemName: "circle.slash.fill")
            disableLabel.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_disable.rawValue)
           
            
            setupSetDefualtButtonEvent(clickable: false)
            setupDisableAccountEvent(clickable: false)
            
        }else{
           
            if(isInactive){
                
                //default icon
                setDefaultIcon.tintColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
                setDefaultLabel.textColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
                disableIcon.tintColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
                disableLabel.textColor = DefaultAppearance.shared.secondaryLabelColor.withAlphaComponent(0.3)
//                disableIcon.image = UIImage(systemName: "circle.slash.fill")
                disableLabel.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_disable.rawValue)
                
                
                disableIcon.tintColor = DefaultAppearance.shared.primaryColor
                disableIcon.image = UIImage(systemName: "checkmark.circle.fill")
                disableLabel.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_enable.rawValue)
                disableLabel.textColor = DefaultAppearance.shared.primaryColor
                //action to enable
                setupEnableAccountEvent()
                
                
                // action on set default
                setupSetDefualtButtonEvent(clickable: false)
                
                
            }else{
                
                setDefaultLabel.textColor = DefaultAppearance.shared.primaryColor
                setDefaultIcon.tintColor = DefaultAppearance.shared.primaryColor
                
                // action on set default
                setupSetDefualtButtonEvent(clickable: true)
                
                
                disableIcon.tintColor = DefaultAppearance.shared.dangerColor
//                disableIcon.image = UIImage(systemName: "circle.slash.fill")
                disableLabel.text = B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_disable.rawValue)
                disableLabel.textColor = DefaultAppearance.shared.dangerColor
                //action to inactive
                setupDisableAccountEvent(clickable: true)
                
            }
        }
    }
    
    // MARK: - Constraint Adjustment Methods
    private func adjustTopupButtonSpacing() {
        // Remove any existing custom constraint
        if let existingConstraint = topupButtonTopConstraint {
            existingConstraint.isActive = false
            topupButtonTopConstraint = nil
        }
        
        // Ensure topup button uses Auto Layout
        topupButton.translatesAutoresizingMaskIntoConstraints = false
        
        // Remove any existing top constraints for topupButton to avoid conflicts
        let constraintsToRemove = view.constraints.filter { constraint in
            (constraint.firstItem as? UIView == topupButton && constraint.firstAttribute == .top) ||
            (constraint.secondItem as? UIView == topupButton && constraint.secondAttribute == .top)
        }
        
        // Also check superview constraints
        if let superview = topupButton.superview {
            let superviewConstraintsToRemove = superview.constraints.filter { constraint in
                (constraint.firstItem as? UIView == topupButton && constraint.firstAttribute == .top) ||
                (constraint.secondItem as? UIView == topupButton && constraint.secondAttribute == .top)
            }
            NSLayoutConstraint.deactivate(superviewConstraintsToRemove)
        }
        
        NSLayoutConstraint.deactivate(constraintsToRemove)
        
        // Create new constraint: topupButton.top = contentContainer.bottom + 20
        topupButtonTopConstraint = topupButton.topAnchor.constraint(
            equalTo: contentContainer.bottomAnchor,
            constant: 20
        )
        topupButtonTopConstraint?.priority = UILayoutPriority(1000) // Higher priority
        topupButtonTopConstraint?.isActive = true
        
        // Force layout update
        view.setNeedsLayout()
        view.layoutIfNeeded()
    }
    
    private func resetTopupButtonSpacing() {
        // Remove the custom constraint to revert to storyboard constraints
        if let existingConstraint = topupButtonTopConstraint {
            existingConstraint.isActive = false
            topupButtonTopConstraint = nil
        }
        
        // Force layout update to restore original storyboard constraints
        view.setNeedsLayout()
        view.layoutIfNeeded()
    }
    
    func topUpButtonAppearance(){
        topupButton.tintColor = DefaultAppearance.shared.primaryColor
//        topupButton.setTitleColor(DefaultAppearance.shared.onPrimaryColor, for: .normal)
//        topupButton.setTitle(B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_top_up.rawValue), for: .normal)
//        topupButton.titleLabel?.font = FontManager.shared.boldFont(forLanguage: language , size: FixFontSize.buttonText)
        
        let topupAttributedTitle = NSAttributedString(
            string: B24PaymentSdkHelper.localized(AccountDetailLocalizedKeys.detail_top_up.rawValue),
                    attributes: [
                        .font: FontManager.shared.regularFont(forLanguage: self.language,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.buttonText),
                        .foregroundColor: DefaultAppearance.shared.onPrimaryColor
                    ]
                )
        self.topupButton.setAttributedTitle(topupAttributedTitle, for: .normal)
    }
    
    // MARK: - Bottom Sheet Properties
    var transactionBottomSheet: TransactionsView?
    private var initialBottomSheetY: CGFloat = 0
    private var currentDetentState: BottomSheetDetent = .medium
    private var isHandlingBottomSheetGesture: Bool = false
    private var lastHapticDetentState: BottomSheetDetent?
    
    enum BottomSheetDetent {
        case medium
        case large
        
        func height(for screenHeight: CGFloat, paymentMethodCount: Int = 1) -> CGFloat {
            switch self {
            case .medium:
                // Increase medium height when there's only one payment method
                return paymentMethodCount == 1 ? screenHeight * 0.48 : screenHeight * 0.4
            case .large:
                // Slightly increase large height when there's only one payment method
                return paymentMethodCount == 1 ? screenHeight * 0.90 : screenHeight * 0.90
            }
        }
        
        func yPosition(for screenHeight: CGFloat, paymentMethodCount: Int = 1) -> CGFloat {
            return screenHeight - height(for: screenHeight, paymentMethodCount: paymentMethodCount)
        }
    }
    
//    func showBottomSheet() {
//        // Instantiate your BottomSheetViewController
//        let storyboard = UIStoryboard(name: "InstantPaymentMethodView", bundle: B24PaymentSdkHelper.frameworkBundle())
//            guard let bottomSheetVC = storyboard.instantiateViewController(withIdentifier: "TransactionsView") as? TransactionsView else { return }
//        presentLegacyBottomSheet(bottomSheetVC)
//
//    }

    func showBottomSheet() {
        
        dismissBottomSheet()
        
        let storyboard = UIStoryboard(name: "InstantPaymentMethodView", bundle: B24PaymentSdkHelper.frameworkBundle())
        guard let bottomSheetVC = storyboard.instantiateViewController(withIdentifier: "TransactionsView") as? TransactionsView else { return }
    
        transactionBottomSheet = bottomSheetVC

        // Pass the account ID
        transactionBottomSheet?.accountId = self.paymentMethodId
        
        // Set delegate for Android-like scroll-to-collapse behavior
        transactionBottomSheet?.delegate = self
    
        presentDraggableBottomSheet(transactionBottomSheet!)
        // Refresh collection insets once the view has laid out at its initial medium height
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.1) { [weak self] in
            self?.transactionBottomSheet?.refreshInsets()
        }
            
    }
    
    func dismissBottomSheet(){
        // If there's a bottom sheet and it's currently presented, dismiss it
        guard let bottomSheet = transactionBottomSheet,
              bottomSheet.isViewLoaded,
              bottomSheet.view.superview != nil else { return }
        
        UIView.animate(withDuration: 0.3, animations: {
            bottomSheet.view.frame.origin.y = self.view.bounds.height
        }, completion: { _ in
            bottomSheet.view.removeFromSuperview()
            bottomSheet.removeFromParent()
            self.transactionBottomSheet = nil
        })
    }
    
   
    
    
    private func presentDraggableBottomSheet(_ bottomSheetVC: TransactionsView) {
        // Add as child view controller
        addChild(bottomSheetVC)
        
        let screenHeight = view.bounds.height
        let mediumHeight = BottomSheetDetent.medium.height(for: screenHeight, paymentMethodCount: paymentMethodCount)
        let largeHeight = BottomSheetDetent.large.height(for: screenHeight, paymentMethodCount: paymentMethodCount)
        
        // Configure bottom sheet view
        bottomSheetVC.view.frame = CGRect(
            x: 0,
            y: screenHeight,
            width: view.bounds.width,
            height: largeHeight
        )
        
        // Add rounded corners
        bottomSheetVC.view.layer.cornerRadius = 16
        bottomSheetVC.view.layer.masksToBounds = true
        bottomSheetVC.view.layer.maskedCorners = [.layerMinXMinYCorner, .layerMaxXMinYCorner]
        
        // Add shadow
        bottomSheetVC.view.layer.shadowColor = UIColor.black.cgColor
        bottomSheetVC.view.layer.shadowOpacity = 0.1
        bottomSheetVC.view.layer.shadowOffset = CGSize(width: 0, height: -2)
        bottomSheetVC.view.layer.shadowRadius = 8
        
        // Add to main view
        view.addSubview(bottomSheetVC.view)
        bottomSheetVC.didMove(toParent: self)
        
        // Add gesture recognizers
        setupBottomSheetGestures(for: bottomSheetVC.view)
        
        // Animate to medium detent
        currentDetentState = .medium
        animateToDetent(.medium, animated: true)
        
        // Set initial Android-like scroll behavior (collapsed = no scrolling)
        transactionBottomSheet?.setBottomSheetState(collapsed: true)
    }

    // MARK: - Gesture Handling

    @objc private func handleBottomSheetPan(_ gesture: UIPanGestureRecognizer) {
        guard let bottomSheetView = gesture.view else { return }
        
        let screenHeight = view.bounds.height
        let translation = gesture.translation(in: view)
        let velocity = gesture.velocity(in: view)
        
        // Check if we should handle this gesture or let the collection view handle it
        let shouldHandle = shouldHandleBottomSheetGesture(gesture: gesture, translation: translation)
        
        switch gesture.state {
        case .began:
            isHandlingBottomSheetGesture = shouldHandle
            if isHandlingBottomSheetGesture {
                initialBottomSheetY = bottomSheetView.frame.origin.y
                lastHapticDetentState = currentDetentState
            }
            
        case .changed:
            // Only continue handling if we decided to handle it at the beginning
            if !isHandlingBottomSheetGesture {
                return
            }
            
            let newY = initialBottomSheetY + translation.y
            let minY = BottomSheetDetent.large.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
            let maxY = BottomSheetDetent.medium.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
            
            // Add rubber band effect for smoother scrolling
            let constrainedY = rubberBandClamp(value: newY, minValue: minY, maxValue: maxY)
            let newHeight = screenHeight - constrainedY
            
            // Use CATransaction to disable implicit animations for smoother dragging
            CATransaction.begin()
            CATransaction.setDisableActions(true)
            
            bottomSheetView.frame = CGRect(
                x: 0,
                y: constrainedY,
                width: view.bounds.width,
                height: newHeight
            )
            
            CATransaction.commit()
            
            // Update grip appearance based on expansion
            updateGripAppearance(for: constrainedY, screenHeight: screenHeight)
            
            // Android-like haptic feedback when crossing detent thresholds
            checkAndTriggerHapticFeedback(for: constrainedY, screenHeight: screenHeight)
            
        case .ended:
            if isHandlingBottomSheetGesture {
                // Update insets once when gesture ends for final layout adjustment
                DispatchQueue.main.async {
                    self.transactionBottomSheet?.refreshInsets()
                }
                // Use enhanced velocity calculation for more natural physics
                let naturalVelocity = calculateNaturalVelocity(from: gesture)
                handlePanGestureEnded(velocity: naturalVelocity, currentY: bottomSheetView.frame.origin.y)
            }
            isHandlingBottomSheetGesture = false
            lastHapticDetentState = nil
            
        case .cancelled, .failed:
            isHandlingBottomSheetGesture = false
            lastHapticDetentState = nil
            
        default:
            break
        }
    }
    
    // MARK: - Haptic Feedback
    private func checkAndTriggerHapticFeedback(for yPosition: CGFloat, screenHeight: CGFloat) {
        let mediumY = BottomSheetDetent.medium.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
        let largeY = BottomSheetDetent.large.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
        
        // Determine which detent we're closest to
        let currentDetentForPosition: BottomSheetDetent
        let midPoint = (mediumY + largeY) / 2
        currentDetentForPosition = yPosition < midPoint ? .large : .medium
        
        // Trigger haptic feedback when crossing into a new detent region
        if let lastDetent = lastHapticDetentState, lastDetent != currentDetentForPosition {
            triggerHapticFeedback()
            lastHapticDetentState = currentDetentForPosition
            
            // Android-like behavior: Update scroll state immediately when crossing threshold
            // This provides immediate feedback during dragging
            let isCollapsed = (currentDetentForPosition == .medium)
            transactionBottomSheet?.setBottomSheetState(collapsed: isCollapsed)
        }
    }
    
    private func triggerHapticFeedback() {
        // Android-like subtle haptic feedback
        if #available(iOS 13.0, *) {
            let impactFeedback = UIImpactFeedbackGenerator(style: .soft)
            impactFeedback.impactOccurred(intensity: 0.6)
        } else {
            let impactFeedback = UIImpactFeedbackGenerator(style: .light)
            impactFeedback.impactOccurred()
        }
    }
    
    private func shouldHandleBottomSheetGesture(gesture: UIPanGestureRecognizer, translation: CGPoint) -> Bool {
        guard let bottomSheetView = transactionBottomSheet?.view,
              let collectionView = transactionBottomSheet?.transactionCollectionView else {
            return true
        }
        
        let location = gesture.location(in: bottomSheetView)
        let gripAreaHeight: CGFloat = 60
        
        // ZONE 1: Grip Area (top 60 points) - Always handle bottom sheet dragging
        if location.y <= gripAreaHeight {
            return true
        }
        
        // ANDROID-LIKE BEHAVIOR: In medium state, ALL gestures should expand the sheet
        // Content scrolling is disabled when collapsed
        if currentDetentState == .medium {
            return true // All touches in medium state expand the bottom sheet
        }
        
        // When in large state, implement Android-like scroll behavior
        let contentOffset = collectionView.contentOffset.y
        let contentHeight = collectionView.contentSize.height
        let frameHeight = collectionView.frame.height
        
        // Use larger thresholds for more stable behavior
        let topThreshold: CGFloat = 15
        let bottomThreshold: CGFloat = 15
        
        // ZONE 2: Collapse Zone - Handle bottom sheet gestures when:
        // - Swiping down to collapse AND at the very top of content (Android behavior)
        if translation.y > 0 && contentOffset <= topThreshold {
            return true
        }
        
        // ZONE 3: Expand Zone - Handle bottom sheet gestures when:
        // - Swiping up AND at the bottom of content (for over-scroll expansion)
        if translation.y < 0 {
            let isAtBottom = (contentOffset + frameHeight >= contentHeight - bottomThreshold)
            
            // Only expand when at bottom of content (Android over-scroll behavior)
            if isAtBottom {
                return true
            }
        }
        
        // ZONE 4: Content Scroll Zone - Let collection view handle scrolling in large state
        return false
    }
    
    private func setupBottomSheetGestures(for bottomSheetView: UIView) {
        // Pan gesture for dragging
        let panGesture = UIPanGestureRecognizer(target: self, action: #selector(handleBottomSheetPan(_:)))
        panGesture.delegate = self
        bottomSheetView.addGestureRecognizer(panGesture)
        
        // Tap gesture to dismiss when tapping outside (configurable)
        if shouldDismissOnOutsideTap {
            let tapGesture = UITapGestureRecognizer(target: self, action: #selector(handleBackgroundTap(_:)))
            tapGesture.delegate = self
            view.addGestureRecognizer(tapGesture)
        }
    }
    
    @objc private func handleBackgroundTap(_ gesture: UITapGestureRecognizer) {
        let location = gesture.location(in: view)
        
        // Only dismiss if tap is outside bottom sheet
        if let bottomSheetView = transactionBottomSheet?.view,
           !bottomSheetView.frame.contains(location) {
            dismissBottomSheet()
        }
    }
    
    private func handlePanGestureEnded(velocity: CGFloat, currentY: CGFloat) {
        let screenHeight = view.bounds.height
        let mediumY = BottomSheetDetent.medium.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
        let largeY = BottomSheetDetent.large.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
        
        let targetDetent: BottomSheetDetent
        
        // Android-like velocity thresholds for more responsive feel
        let velocityThreshold: CGFloat = 200
        let strongVelocityThreshold: CGFloat = 800
        
        // NO DISMISS - Only collapse to medium or expand to large
        // Determine target based on velocity and position with Android-like behavior
        if abs(velocity) > strongVelocityThreshold {
            // Strong velocity - follow the direction but only between medium and large
            targetDetent = velocity > 0 ? .medium : .large
        } else if abs(velocity) > velocityThreshold {
            // Medium velocity - bias towards the direction but consider position
            let midPoint = (mediumY + largeY) / 2
            if velocity > 0 {
                // Swiping down - prefer medium but consider position
                targetDetent = currentY > midPoint * 0.8 ? .medium : .large
            } else {
                // Swiping up - prefer large but consider position
                targetDetent = currentY < midPoint * 1.2 ? .large : .medium
            }
        } else {
            // Low velocity - determine based on position with hysteresis
            let midPoint = (mediumY + largeY) / 2
            let hysteresis: CGFloat = 40 // Larger hysteresis for better UX
            
            if currentDetentState == .large {
                targetDetent = currentY > (midPoint + hysteresis) ? .medium : .large
            } else {
                targetDetent = currentY < (midPoint - hysteresis) ? .large : .medium
            }
        }
        
        currentDetentState = targetDetent
        animateToDetentWithVelocity(targetDetent, initialVelocity: velocity)
    }
    
    private func animateToDetent(_ detent: BottomSheetDetent, animated: Bool = true) {
        animateToDetentWithVelocity(detent, initialVelocity: 0, animated: animated)
    }
    
    private func animateToDetentWithVelocity(_ detent: BottomSheetDetent, initialVelocity: CGFloat = 0, animated: Bool = true) {
        guard let bottomSheetView = transactionBottomSheet?.view else { return }
        
        let screenHeight = view.bounds.height
        let targetY = detent.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
        let targetHeight = detent.height(for: screenHeight, paymentMethodCount: paymentMethodCount)
        
        let animations = {
            bottomSheetView.frame = CGRect(
                x: 0,
                y: targetY,
                width: self.view.bounds.width,
                height: targetHeight
            )
            self.updateGripAppearance(for: targetY, screenHeight: screenHeight)
        }
        
        let completion: (Bool) -> Void = { _ in
            self.transactionBottomSheet?.refreshInsets()
            
            // Update Android-like scroll behavior based on detent state
            let isCollapsed = (detent == .medium)
            self.transactionBottomSheet?.setBottomSheetState(collapsed: isCollapsed)
        }
        
        if animated {
            // Calculate duration based on distance and velocity for natural feel
            let currentY = bottomSheetView.frame.origin.y
            let distance = abs(targetY - currentY)
            let baseVelocity: CGFloat = max(abs(initialVelocity), 300.0 as CGFloat)
            let duration = min(max(distance / baseVelocity, 0.25 as CGFloat), 0.8 as CGFloat)
            
            // Enhanced Android-like spring animation with consistent feel for both expand/collapse
            let dampingRatio: CGFloat = detent == .large ? 0.82 : 0.88 // Slightly different damping for expand vs collapse
            let springAnimator = UIViewPropertyAnimator(duration: TimeInterval(duration), dampingRatio: dampingRatio) {
                animations()
            }
            
            // Enhanced velocity continuation for smoother gesture flow
            if abs(initialVelocity) > 30.0 as CGFloat { // Reduced threshold for better responsiveness
                // Add initial spring velocity for more natural continuation
                springAnimator.addAnimations({
                    // Additional subtle bounce for enhanced feedback
                    let bounceScale = detent == .large ? 1.002 : 0.998
                    bottomSheetView.transform = CGAffineTransform(scaleX: bounceScale, y: bounceScale)
                }, delayFactor: 0.3)
                
                springAnimator.addAnimations({
                    bottomSheetView.transform = .identity
                }, delayFactor: 0.7)
            }
            
            springAnimator.addCompletion { _ in
                completion(true)
            }
            
            springAnimator.startAnimation()
        } else {
            animations()
            completion(true)
        }
    }
    
    private func dismissBottomSheetWithAnimation() {
        guard let bottomSheet = transactionBottomSheet,
              bottomSheet.isViewLoaded,
              bottomSheet.view.superview != nil else { return }
        
        let screenHeight = view.bounds.height
        
        // Remove background dim effect immediately
        removeBackgroundDimEffect()
        
        // Android-like dismiss animation with fade and slide
        let slideAnimation = UIViewPropertyAnimator(duration: 0.4, dampingRatio: 0.9) {
            bottomSheet.view.frame.origin.y = screenHeight + 50
            bottomSheet.view.alpha = 0.0
            
            // Optional: Add slight scale animation for more polish
            bottomSheet.view.transform = CGAffineTransform(scaleX: 0.95, y: 0.95)
        }
        
        slideAnimation.addCompletion { _ in
            bottomSheet.view.removeFromSuperview()
            bottomSheet.removeFromParent()
            self.transactionBottomSheet = nil
        }
        
        slideAnimation.startAnimation()
    }
    
    private func rubberBandClamp(value: CGFloat, minValue: CGFloat, maxValue: CGFloat) -> CGFloat {
        // Android-like rubber band effect with progressive resistance
        if value < minValue {
            let diff = minValue - value
            let maxOverscroll: CGFloat = 100.0 // Maximum allowed overscroll upward
            let normalizedDiff = min(diff, maxOverscroll) / maxOverscroll
            
            // Progressive resistance curve - stronger resistance as you pull further
            let resistance = CGFloat(1.0 - pow(Double(normalizedDiff), 0.6))
            let rubberBandedDiff = diff * resistance * 0.25
            
            return minValue - rubberBandedDiff
        } else if value > maxValue {
            let diff = value - maxValue
            // Reduced overscroll when pulling down to prevent impression of dismissal
            let maxOverscroll: CGFloat = 80.0 // Reduced from 150 to prevent dismiss impression
            let normalizedDiff = min(diff, maxOverscroll) / maxOverscroll
            
            // Stronger resistance when pulling down to discourage excessive dragging
            let resistance = CGFloat(1.0 - pow(Double(normalizedDiff), 0.8)) // Increased from 0.7
            let rubberBandedDiff = diff * resistance * 0.3 // Reduced from 0.4
            
            return maxValue + rubberBandedDiff
        }
        return value
    }
    
    // MARK: - Enhanced Physics Simulation
    private func calculateNaturalVelocity(from gesture: UIPanGestureRecognizer) -> CGFloat {
        let velocity = gesture.velocity(in: view).y
        
        // Apply velocity decay based on distance from bounds for more natural feel
        guard let bottomSheetView = transactionBottomSheet?.view else { return velocity }
        
        let screenHeight = view.bounds.height
        let currentY = bottomSheetView.frame.origin.y
        let minY = BottomSheetDetent.large.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
        let maxY = BottomSheetDetent.medium.yPosition(for: screenHeight, paymentMethodCount: paymentMethodCount)
        
        // Reduce velocity when near rubber band zones for smoother transitions
        if currentY < minY {
            let overscrollRatio = (minY - currentY) / 100.0
            let dampingFactor = max(0.3 as CGFloat, 1.0 - overscrollRatio * 0.7)
            return velocity * dampingFactor
        } else if currentY > maxY {
            let overscrollRatio = (currentY - maxY) / 150.0
            let dampingFactor = max(0.4 as CGFloat, 1.0 - overscrollRatio * 0.6)
            return velocity * dampingFactor
        }
        
        return velocity
    }
    
    private func updateGripAppearance(for yPosition: CGFloat, screenHeight: CGFloat) {
        let mediumHeight = BottomSheetDetent.medium.height(for: screenHeight, paymentMethodCount: paymentMethodCount)
        let largeHeight = BottomSheetDetent.large.height(for: screenHeight, paymentMethodCount: paymentMethodCount)
        let currentHeight = screenHeight - yPosition
        
        let expansionFraction = (currentHeight - mediumHeight) / (largeHeight - mediumHeight)
        let clampedFraction = max(0.0 as CGFloat, min(1.0 as CGFloat, expansionFraction))
        
        // Update the grip appearance in the transaction view
        transactionBottomSheet?.updateGripAppearance(progress: clampedFraction)
        
        // Android-like dynamic visual feedback
        updateBottomSheetVisualFeedback(expansionProgress: clampedFraction)
    }
    
    private func updateBottomSheetVisualFeedback(expansionProgress: CGFloat) {
        guard let bottomSheetView = transactionBottomSheet?.view else { return }
        
        // Dynamic shadow intensity - stronger shadow when more expanded
        let baseShadowOpacity: Float = 0.1
        let maxShadowOpacity: Float = 0.25
        let shadowOpacity = baseShadowOpacity + (maxShadowOpacity - baseShadowOpacity) * Float(expansionProgress)
        
        let baseShadowRadius: CGFloat = 8
        let maxShadowRadius: CGFloat = 20
        let shadowRadius = baseShadowRadius + (maxShadowRadius - baseShadowRadius) * expansionProgress
        
        // Apply shadow changes smoothly
        CATransaction.begin()
        CATransaction.setDisableActions(true)
        
        bottomSheetView.layer.shadowOpacity = shadowOpacity
        bottomSheetView.layer.shadowRadius = shadowRadius
        bottomSheetView.layer.shadowOffset = CGSize(width: 0, height: -2 - (4 * expansionProgress))
        
        // Subtle corner radius adjustment for more dynamic feel
        let baseCornerRadius: CGFloat = 16
        let maxCornerRadius: CGFloat = 20
        let cornerRadius = baseCornerRadius + (maxCornerRadius - baseCornerRadius) * expansionProgress
        bottomSheetView.layer.cornerRadius = cornerRadius
        
        CATransaction.commit()
        
        // Optional: Add subtle background dim effect when fully expanded
        if expansionProgress > 0.8 as CGFloat {
            addBackgroundDimEffect(alpha: (expansionProgress - 0.8 as CGFloat) * 0.2 as CGFloat)
        } else {
            removeBackgroundDimEffect()
        }
    }
    
    private func addBackgroundDimEffect(alpha: CGFloat) {
        // Add a subtle dim overlay behind the bottom sheet when expanded
        if view.viewWithTag(999) == nil {
            let dimView = UIView()
            dimView.tag = 999
            dimView.backgroundColor = UIColor.black
            dimView.alpha = 0
            
            view.insertSubview(dimView, belowSubview: transactionBottomSheet!.view)
            dimView.translatesAutoresizingMaskIntoConstraints = false
            NSLayoutConstraint.activate([
                dimView.topAnchor.constraint(equalTo: view.topAnchor),
                dimView.leadingAnchor.constraint(equalTo: view.leadingAnchor),
                dimView.trailingAnchor.constraint(equalTo: view.trailingAnchor),
                dimView.bottomAnchor.constraint(equalTo: view.bottomAnchor)
            ])
        }
        
        view.viewWithTag(999)?.alpha = alpha
    }
    
    private func removeBackgroundDimEffect() {
        UIView.animate(withDuration: 0.2) {
            self.view.viewWithTag(999)?.alpha = 0
        } completion: { _ in
            self.view.viewWithTag(999)?.removeFromSuperview()
        }
    }
    
    
    // MARK:request detail
    
    private func accountDetail(){
        guard NetworkMonitor.shared.isConnected else {
            if !isWaitingForConnection {
                isWaitingForConnection = true
                B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                    guard let self = self else { return }
                    self.isWaitingForConnection = false
                    B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                }
            }
            return
        }
    
        startLoading()
        
        AF.request(WalletRouter.instantPaymentMethodDetail(id: paymentMethodId))
            .validate().responseData{
                [self](response) in
                switch response.result{
                case .success(let data):
                   
                    do{
                        let decodeData = try JSONDecoder().decode(ApiResponse<DataAccountDetail>.self,from: data)
                        if decodeData.code == StatusCode.succcess.rawValue{
                            
                            emptyImage.isHidden = true
                            emptyLabel.isHidden = true
                        
                            stopLoading()
                            
                            minHeight = 0.4
                            
                            dataAccountDetail = decodeData.data
                            
                            isDefault = dataAccountDetail.isDefault
                            isInactive = dataAccountDetail.inactive
                            accountNo = dataAccountDetail.accountNo
                            DispatchQueue.main.async {
                                self.applyToolBar()
                                self.showBottomSheet()
                            }
                            headerAppearance(data: dataAccountDetail)
                            contentAppearance(data: dataAccountDetail)
    
                            buttonAppearance(data: dataAccountDetail)
                            
                            topUpButtonAppearance()
                
                            
                            //set share Preference
                            setSharePref(pmId: dataAccountDetail.id, walletName: dataAccountDetail.accountName, walletNo: dataAccountDetail.accountNo, amount: dataAccountDetail.amountDisplay, currency: dataAccountDetail.currency)
                            
                            dismissBottomSheet()
                            
                        }else{
                            
                            progressIndicatorMain.stopAnimating()
                            progressIndicatorMain.isHidden = true
                            
                            emptyImage.isHidden = false
                            emptyLabel.isHidden = false
                            
                            emptyImage.image = UIImage(systemName: emptyIconName)
                            emptyImage.tintColor = DefaultAppearance.shared.warningColor
                            emptyLabel.text = B24PaymentSdkHelper.localized(PaymentMethodWalletLocalizedKeys.emptyText.rawValue)
                            emptyLabel.textColor = DefaultAppearance.shared.primaryLabelColor
                            emptyLabel.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.contentText)
                            
                            
                            let message = language == "en" ? decodeData.message : decodeData.messageKh
                            
                            B24PaymentSdkHelper.errorSnackbar(view: view,
                                                              message: message )
                        }
                        
                    }catch{
                        print("Decode Error: \(error)")
                    }
                case .failure(let error):
                   
                    
                    print("==>\(error)")
                    
                    progressIndicatorMain.stopAnimating()
                    progressIndicatorMain.isHidden = true
                    
                    if !self.isWaitingForConnection {
                        self.isWaitingForConnection = true
                        B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                            guard let self = self else { return }
                            self.isWaitingForConnection = false
                            B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                        }
                    } else {
                        emptyImage.isHidden = false
                        emptyLabel.isHidden = false
                        
                        
                        emptyImage.image = UIImage(systemName: emptyIconName)
                        emptyImage.tintColor = DefaultAppearance.shared.warningColor
                        emptyLabel.text = B24PaymentSdkHelper.localized(PaymentMethodWalletLocalizedKeys.emptyText.rawValue)
                        emptyLabel.textColor = DefaultAppearance.shared.primaryLabelColor
                        emptyLabel.font = FontManager.shared.regularFont(forLanguage: language ,fontFromconfig: DefaultAppearance.shared.fontEnglish, size: FixFontSize.contentText)
                        
                        B24PaymentSdkHelper.errorSnackbar(
                                        view: view,
                                        message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                                                            forBottomSheet: false
                                                    )
                    }
                    
                }
            }
        
    }
    
    // MARK: Set Account Default
    func setAccountDefult(){
        let payload = SetDefaultPayload(
            id: paymentMethodId)
        
        do {
            let jsonData = try JSONEncoder().encode(payload)
            guard let jsonString = String(data: jsonData, encoding: .utf8) else {
                print( "Failed to create payload")
                return
            }

            let encryptedText = try EncryptionHelper.encryptHMACAES(
                plainText: jsonString,
                secretKey: SDKVariableSetting.walletSecretKey
            )
           
            sendSetDetaultAccount(encrypt: encryptedText)

        } catch {
            print("Encryption failed: \(error.localizedDescription)")
        }
    }
    // MARK: request set detautl account
    func sendSetDetaultAccount(encrypt: String){
        guard NetworkMonitor.shared.isConnected else {
            if !isWaitingForConnection {
                isWaitingForConnection = true
                B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                    guard let self = self else { return }
                    self.isWaitingForConnection = false
                    B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                }
            }
            return
        }
     
        let progressDiglog = LoadingView.show(in: self)
        
        AF.request(WalletRouter.setDetault(encrypted: encrypt))
            .validate().responseData{
               response in
                switch response.result{
                    case .success(let data):
                    progressDiglog.dismiss()
                        do{
                            let decodedData = try JSONDecoder().decode(ApiResponse<WalletResponseData>.self, from: data)
                            
                            if decodedData.code == StatusCode.succcess.rawValue{
                                DispatchQueue.main.async {
                                    self.isDefault = !self.isDefault

                                    self.headerAppearance(data: self.dataAccountDetail)
                                    self.buttonAppearance(data: self.dataAccountDetail)
                                }
                            
                            }else{
                                let message = self.language == "en" ? decodedData.message : decodedData.messageKh
                                
                                B24PaymentSdkHelper.errorSnackbar(view: self.view,
                                                                  message: message )
                            }
                            
                            print("===========>\(decodedData)")
                        }catch{
                            print("Decode Error: \(error)")
                       
                        }
                    
                    case.failure(let error):
                        progressDiglog.dismiss()
                        
                        if !self.isWaitingForConnection {
                            self.isWaitingForConnection = true
                            B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                                guard let self = self else { return }
                                self.isWaitingForConnection = false
                                B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                            }
                        } else {
                            B24PaymentSdkHelper.errorSnackbar(
                                view:self.view,
                                            message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                                                                forBottomSheet: false
                                                        )
                        }
                        print("ERROR \(error)")
                }
            }
    }
    
    
    func setupSetDefualtButtonEvent(clickable:Bool){
        setDefaultButtonContainer.isUserInteractionEnabled = clickable
        let tabAction = UITapGestureRecognizer(target: self, action: #selector(setDefaultTap))
        setDefaultButtonContainer.addGestureRecognizer(tabAction)
    }
    
    @objc func setDefaultTap(){
        
        let dialog = CustomAlertView()
        
        dialog.showCustomAlert(from: self, isDefaultIcon: true, message: B24PaymentSdkHelper.localized(DialogLocalizedKeys.set_default.rawValue)) {
            self.setAccountDefult()
        } cancelHandler: {
            
        }

    }
    
    func setupDisableAccountEvent(clickable:Bool){
        
        disableContainer.isUserInteractionEnabled = clickable
        let tabAction = UITapGestureRecognizer(target: self, action: #selector(disableAccount))
        disableContainer.addGestureRecognizer(tabAction)
    }
    
    
    @objc func disableAccount(){
        
        let dialog = WarningAlertView()
        dialog.showWarningAlert(from: self) {
            self.updateAccountStatus(inactive: true)
        } cancelHandler: {
            
        }

        
    }
    
    func setupEnableAccountEvent(){
        
        disableContainer.isUserInteractionEnabled = true
        let tabAction = UITapGestureRecognizer(target: self, action: #selector(enableAccount))
        disableContainer.addGestureRecognizer(tabAction)
    }
    
    
    @objc func enableAccount(){
       let dialog = CustomAlertView()
        dialog.showCustomAlert(from: self, isDefaultIcon: false, message: B24PaymentSdkHelper.localized(DialogLocalizedKeys.enable.rawValue)) {
            self.updateAccountStatus(inactive: false)
        } cancelHandler: {
            
        }

        
    }
    
    func updateAccountStatus(inactive:Bool){
        let payload = UpdateStatus(
            id: paymentMethodId,
            inactive: inactive)
        
        do {
            let jsonData = try JSONEncoder().encode(payload)
            guard let jsonString = String(data: jsonData, encoding: .utf8) else {
                print("Failed to create payload")
                return
            }

            let encryptedText = try EncryptionHelper.encryptHMACAES(
                plainText: jsonString,
                secretKey: SDKVariableSetting.walletSecretKey
            )
           
            sendUpdateStatus(encrypt: encryptedText,inactive: inactive)

        } catch {
            print("Encryption failed: \(error.localizedDescription)")
        }
        
        
    }
    
    //MARK: request update status
    func sendUpdateStatus(encrypt:String,inactive:Bool){
        guard NetworkMonitor.shared.isConnected else {
            if !isWaitingForConnection {
                isWaitingForConnection = true
                B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                    guard let self = self else { return }
                    self.isWaitingForConnection = false
                    B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                }
            }
            return
        }
        let progressDiglog = LoadingView.show(in: self)
        
        AF.request(WalletRouter.updateStatus(encrypted: encrypt))
            .validate().responseData{
               response in
                switch response.result{
                    case .success(let data):
                    progressDiglog.dismiss()
                        do{
                            let decodedData = try JSONDecoder().decode(ApiResponse<WalletResponseData>.self, from: data)
                            
                            if decodedData.code == StatusCode.succcess.rawValue{
                                DispatchQueue.main.async {
                                    //self.isDefault = false
                                    self.isInactive = inactive
                                    
                                    self.headerAppearance(data: self.dataAccountDetail)
                                    self.buttonAppearance(data: self.dataAccountDetail)
                                }
                            }else{
                                let message = self.language == "en" ? decodedData.message : decodedData.messageKh
                                
                                B24PaymentSdkHelper.errorSnackbar(view: self.view,
                                                                  message: message )
                            }
                            
                
                        }catch{
                            print("Decode Error: \(error)")
                       
                        }
                    
                    case.failure(let error):
                        progressDiglog.dismiss()
                        
                        if !self.isWaitingForConnection {
                            self.isWaitingForConnection = true
                            B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                                guard let self = self else { return }
                                self.isWaitingForConnection = false
                                B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                            }
                        } else {
                            B24PaymentSdkHelper.errorSnackbar(
                                view:self.view,
                                            message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                                                                forBottomSheet: false
                                                        )
                        }
                        print("ERROR \(error)")
                }
            }
    }
    
    
    
    func setupModifyName(){
        iconEdit.isUserInteractionEnabled = true
        let tabAction = UITapGestureRecognizer(target: self, action: #selector(modifyNameTapped))
        iconEdit.addGestureRecognizer(tabAction)
    }
    
    @objc func modifyNameTapped(){
        
        if isInactive{
            let dialog = WarningAlertView()
            dialog.showWarningTopupAlert(from: self) {
                self.updateAccountStatus(inactive: false)
            } cancelHandler: {
                
            }

        }else{
//            let dialog = EditDialog()
//            dialog.showEditDialog(from: self) { inputText in
//                self.modifyName(name: inputText)
//            } cancelHandler: {
//                
//            }

            let dialog = EditDialog()
              dialog.showEditDialog(
                  from: self,
                  defaultText: self.lblTitle.text ?? self.dataAccountDetail.accountName  // Pass the current name
              ) { inputText in
                  self.modifyName(name: inputText)
              } cancelHandler: {

              }
        }
    }
    
    func modifyName(name:String){
        
        
        let payload = ModifyNamePayload(id: paymentMethodId, name: name)
        
        do {
            let jsonData = try JSONEncoder().encode(payload)
            guard let jsonString = String(data: jsonData, encoding: .utf8) else {
               print("Failed to create payload")
                return
            }

            let encryptedText = try EncryptionHelper.encryptHMACAES(
                plainText: jsonString,
                secretKey: SDKVariableSetting.walletSecretKey
            )
           
            sendModifyRequest(encrypt: encryptedText,name: name)

        } catch {
            print( "Encryption failed: \(error.localizedDescription)")
        }
    }
    
    //MARK: requst modify name
    func sendModifyRequest(encrypt:String,name:String){
        guard NetworkMonitor.shared.isConnected else {
            if !isWaitingForConnection {
                isWaitingForConnection = true
                B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                    guard let self = self else { return }
                    self.isWaitingForConnection = false
                    B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                }
            }
            return
        }
        let progressDiglog = LoadingView.show(in: self)
        
        AF.request(WalletRouter.modifyWalletName(encrypted: encrypt))
            .validate().responseData{
               response in
                switch response.result{
                    case .success(let data):
                    progressDiglog.dismiss()
                        do{
                            let decodedData = try JSONDecoder().decode(ApiResponse<WalletResponseData>.self, from: data)
                            
                            if decodedData.code == StatusCode.succcess.rawValue{
                                self.lblTitle.text = name
                            }else{
                                let message = self.language == "en" ? decodedData.message : decodedData.messageKh
                                
                                B24PaymentSdkHelper.errorSnackbar(view: self.view,
                                                                  message: message )
                            }
                            
                            print("===========>\(decodedData)")
                        }catch{
                            print("Decode Error: \(error)")
                       
                        }
                    
                    case.failure(let error):
                        progressDiglog.dismiss()
                        
                        if !self.isWaitingForConnection {
                            self.isWaitingForConnection = true
                            B24PaymentSdkHelper.showNoInternetPopup(on: self) { [weak self] in
                                guard let self = self else { return }
                                self.isWaitingForConnection = false
                                B24PaymentSdkHelper.hideNoInternetPopup(from: self, showSuccess: false)
                            }
                        } else {
                            B24PaymentSdkHelper.errorSnackbar(
                                view: self.view,
                                            message: B24PaymentSdkHelper.localized(SnackBarLocalizedKeys.error.rawValue),
                                                                forBottomSheet: false
                                                        )
                        }
                        print("ERROR \(error)")
                }
            }
    }
   
    
    private func setSharePref(pmId:String,walletName:String,walletNo:String,amount:String,currency:String){
        SharedPreferenceManager.set(value: pmId, forKey: SharePrefKey.paymentMethodId.rawValue)
        SharedPreferenceManager.set(value: walletName, forKey: SharePrefKey.walletName.rawValue)
        SharedPreferenceManager.set(value: walletNo, forKey: SharePrefKey.walletNo.rawValue)
        SharedPreferenceManager.set(value: amount, forKey: SharePrefKey.balance.rawValue)
        SharedPreferenceManager.set(value: currency, forKey: SharePrefKey.currency.rawValue)

    }
    
}


// MARK: - Account Detail View Extension
//extension AccountDetailView {
//
//    // MARK: - Setup Methods
//    private func setupLoadingIndicator() {
//        view.addSubview(loadingIndicator)
//        NSLayoutConstraint.activate([
//            loadingIndicator.centerXAnchor.constraint(equalTo: view.centerXAnchor),
//            loadingIndicator.centerYAnchor.constraint(equalTo: view.centerYAnchor)
//        ])
//    }
//
//    private func updateLoadingState() {
//        DispatchQueue.main.async {
//            if self.isLoading {
//                self.loadingIndicator.startAnimating()
//                self.contentContainer.isHidden = true
//            } else {
//                self.loadingIndicator.stopAnimating()
//                self.contentContainer.isHidden = false
//            }
//        }
//    }
    

    // MARK: - API Methods
//    func fetchAccountDetail(accountId: String) {
//        isLoading = true
//
//        let url = APIManager.merchantApiUrl().appendingPathComponent("instantpaymentsdk/payment_method/detail")
//        let parameters: [String: Any] = ["id": accountId]
//
//        AF.request(
//            url,
//            method: .post,
//            parameters: parameters,
//            encoding: JSONEncoding.default,
//            headers: APIManager.initHeader()
//        )
//        .validate()
//        .responseDecodable(of: ApiResponse<DataAccountDetail>.self) { [weak self] response in
//            guard let self = self else { return }
//            self.isLoading = false
//
//            switch response.result {
//            case .success(let apiResponse):
//                self.handleSuccessResponse(apiResponse)
//            case .failure(let error):
//                self.handleError(error)
//            }
//        }
//    }

    // MARK: - Response Handlers
//    private func handleSuccessResponse(_ apiResponse: ApiResponse<DataAccountDetail>) {
//        if apiResponse.code == "SUCCESS" {
//            DispatchQueue.main.async {
//                self.updateUI(with: apiResponse.data)
//            }
//        } else {
//            showError(message: apiResponse.message)
//        }
//    }

//    private func handleError(_ error: AFError) {
//        let errorMessage: String
//        switch error {
//        case .responseValidationFailed(let reason):
//            errorMessage = "Validation failed: \(reason)"
//        case .responseSerializationFailed(let reason):
//            errorMessage = "Failed to process response: \(reason)"
//        case .sessionTaskFailed(let error):
//            errorMessage = "Network error: \(error.localizedDescription)"
//        default:
//            errorMessage = "An unexpected error occurred"
//        }
//        showError(message: errorMessage)
//    }

    // MARK: - UI Updates
//    private func updateUI(with data: DataAccountDetail) {
//        self.dataAccountDetail = data
//
//        headerAppearance(data: data)
//        contentAppearance(data: data)
//        buttonAppearance(data: data)
//
//        // Update logo if available
//        if !data.logo.isEmpty {
//            loadLogo(from: data.logo)
//        }
//    }

//    private func loadLogo(from urlString: String) {
//        guard let url = URL(string: urlString) else { return }
//
//        DispatchQueue.global().async {
//            if let data = try? Data(contentsOf: url),
//               let image = UIImage(data: data) {
//                DispatchQueue.main.async {
//                    self.logo.image = image
//                }
//            }
//        }
//    }

    // MARK: - Error Handling
//    private func showError(message: String) {
//        DispatchQueue.main.async {
//            let alert = UIAlertController(
//                title: "Error",
//                message: message,
//                preferredStyle: .alert
//            )
//            alert.addAction(UIAlertAction(title: "OK", style: .default))
//            self.present(alert, animated: true)
//        }
//    }
//}

// MARK: - View Controller Lifecycle
//extension AccountDetailView {
//    private func setupInitialUI() {
//        view.backgroundColor = DefaultAppearance.shared.screenBgColor
//        getSharePref()
//        B24PaymentSdkHelper.getCurrentLanguage(language: language)
//        showBottomSheet()
//
//        // Setup initial state with default data
//        headerAppearance(data: dataAccountDetail)
//        contentAppearance(data: dataAccountDetail)
//        buttonAppearance(data: dataAccountDetail)
//        topUpButtonAppearance()
//    }
//}

// MARK: - UIGestureRecognizerDelegate
extension AccountDetailView {
    func gestureRecognizer(_ gestureRecognizer: UIGestureRecognizer, shouldReceive touch: UITouch) -> Bool {
        // Additional touch filtering for better gesture handling
        if gestureRecognizer is UIPanGestureRecognizer {
            let location = touch.location(in: view)
            
            // If touch is outside bottom sheet, don't handle it
            if let bottomSheetView = transactionBottomSheet?.view {
                let sheetFrame = bottomSheetView.frame
                if !sheetFrame.contains(location) {
                    return false
                }
            }
        }
        
        // For tap gestures, only handle if it's the background tap
        if gestureRecognizer is UITapGestureRecognizer {
            // Don't handle tap if it's inside the bottom sheet
            if let bottomSheetView = transactionBottomSheet?.view {
                let location = touch.location(in: view)
                return !bottomSheetView.frame.contains(location)
            }
        }
        
        return true
    }
    
    func gestureRecognizer(_ gestureRecognizer: UIGestureRecognizer, shouldRecognizeSimultaneouslyWith otherGestureRecognizer: UIGestureRecognizer) -> Bool {
        // Be more selective about simultaneous recognition to avoid conflicts
        
        // If this is a pan gesture from the bottom sheet
        if gestureRecognizer is UIPanGestureRecognizer && gestureRecognizer.view == transactionBottomSheet?.view {
            // Only allow simultaneous recognition with scroll views when they're different types of gestures
            // or when we're in specific scroll conditions
            if let scrollView = otherGestureRecognizer.view as? UIScrollView {
                let panGesture = gestureRecognizer as! UIPanGestureRecognizer
                let translation = panGesture.translation(in: view)
                
                // Allow simultaneous recognition only if our custom logic says we should handle the gesture
                return shouldHandleBottomSheetGesture(gesture: panGesture, translation: translation)
            }
            return false
        }
        
        // For other gestures, allow simultaneous recognition
        return true
    }
}

// MARK: - TransactionsViewDelegate
extension AccountDetailView {
    func shouldCollapseBottomSheet() {
        // Only collapse if currently in large state
        guard currentDetentState == .large else { return }
        
        // Animate to medium (collapsed) state with enhanced animation
        currentDetentState = .medium
        animateToDetent(.medium, animated: true)
        
        // Ensure the transactions view knows it's being collapsed
        transactionBottomSheet?.setBottomSheetState(collapsed: true)
        
        // Provide haptic feedback for the collapse action
        triggerHapticFeedback()
    }
    
    func shouldExpandBottomSheet() {
        // Only expand if currently in medium state
        guard currentDetentState == .medium else { return }
        
        // Animate to large (expanded) state with enhanced animation
        currentDetentState = .large
        animateToDetent(.large, animated: true)
        
        // Ensure the transactions view knows it's being expanded
        // Use a small delay to ensure animation has started
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.1) {
            self.transactionBottomSheet?.setBottomSheetState(collapsed: false)
        }
        
        // Provide haptic feedback for the expand action
        triggerHapticFeedback()
    }
}
